<?php

namespace DesignGrid;


class BlackdsnBaseAttr {
	/**
	 *
	 * Holds all the render attributes of the element. Used to store data like
	 * the HTML class name and the class value, or HTML element ID name and value.
	 *
	 * @access private
	 *
	 * @var array
	 */
	private $render_attributes = [];


	/**
	 * @param $element
	 * @param  null  $key
	 * @param  null  $value
	 * @param  false  $overwrite
	 *
	 * @return $this
	 */
	public function add_render_attribute( $element, $key = null, $value = null, $overwrite = false ) {
		if ( is_array( $element ) ) {
			foreach ( $element as $element_key => $attributes ) {
				$this->add_render_attribute( $element_key, $attributes, null, $overwrite );
			}

			return $this;
		}

		if ( is_array( $key ) ) {
			foreach ( $key as $attribute_key => $attributes ) {
				$this->add_render_attribute( $element, $attribute_key, $attributes, $overwrite );
			}

			return $this;
		}

		if ( empty( $this->render_attributes[ $element ][ $key ] ) ) {
			$this->render_attributes[ $element ][ $key ] = [];
		}

		settype( $value, 'array' );

		if ( $overwrite ) {
			$this->render_attributes[ $element ][ $key ] = $value;
		} else {
			$this->render_attributes[ $element ][ $key ] = array_merge( $this->render_attributes[ $element ][ $key ],
				$value );
		}

		return $this;
	}


	/**
	 * @param  array  $element
	 * @param  null  $key
	 * @param  null  $value
	 * @param  false  $overwrite
	 *
	 * @return $this
	 */
	public function add_render_duplicate_attribute( array $element, $key = null, $value = null, $overwrite = false ) {
		foreach ( $element as $element_key ):
			$this->add_render_attribute( $element_key, $key, $value, $overwrite );
		endforeach;

		return $this;
	}


	/**
	 * @param  string  $element
	 * @param  string  $key
	 *
	 * @return array|mixed|null
	 */
	public function get_render_attributes( $element = '', $key = '' ) {
		$attributes = $this->render_attributes;

		if ( $element ) {
			if ( ! isset( $attributes[ $element ] ) ) {
				return null;
			}

			$attributes = $attributes[ $element ];

			if ( $key ) {
				if ( ! isset( $attributes[ $key ] ) ) {
					return null;
				}

				$attributes = $attributes[ $key ];
			}
		}

		return $attributes;
	}


	/**
	 * @param $element
	 * @param  null  $key
	 * @param  null  $values
	 */
	public function remove_render_attribute( $element, $key = null, $values = null ) {
		if ( $key && ! isset( $this->render_attributes[ $element ][ $key ] ) ) {
			return;
		}

		if ( $values ) {
			$values = (array) $values;

			$this->render_attributes[ $element ][ $key ] = array_diff( $this->render_attributes[ $element ][ $key ],
				$values );

			return;
		}

		if ( $key ) {
			unset( $this->render_attributes[ $element ][ $key ] );

			return;
		}

		if ( isset( $this->render_attributes[ $element ] ) ) {
			unset( $this->render_attributes[ $element ] );
		}
	}


	/**
	 * @param $element
	 *
	 * @return string
	 */
	public function get_render_attribute_string( $element, $echo = false ) {

		if ( empty( $this->render_attributes[ $element ] ) ) {
			return '';
		}

		$rendered_attributes = [];

		foreach ( $this->render_attributes[ $element ] as $attribute_key => $attribute_values ) {
			if ( is_array( $attribute_values ) ) {
				$attribute_values = implode( ' ', $attribute_values );
			}

			$rendered_attributes[] = sprintf( '%1$s="%2$s"', $attribute_key, esc_attr( $attribute_values ) );
		}
		if ( $echo ) {
			echo implode( ' ', $rendered_attributes );
		} else {
			return implode( ' ', $rendered_attributes );
		}
	}


	public function print_render_attribute_string( $element ) {
		$this->get_render_attribute_string( $element, true ); // XSS ok.
	}


	public function add_link_attributes(
		$element,
		array $url_control,
		$attributes = [],
		$overwrite = false
	): BlackdsnBaseAttr {


		if ( ! empty( $url_control['url'] ) ) {
			$allowed_protocols = array_merge( wp_allowed_protocols(), [ 'skype', 'viber' ] );

			$attributes['href'] = esc_url( $url_control['url'], $allowed_protocols );
		}

		if ( ! empty( $url_control['is_external'] ) ) {
			$attributes['target'] = '_blank';
		}

		if ( ! empty( $url_control['nofollow'] ) ) {
			$attributes['rel'] = 'nofollow';
		}

		if ( ! empty( $url_control['custom_attributes'] ) && class_exists( '\Elementor\Utils' ) ) {
			// Custom URL attributes should come as a string of comma-delimited key|value pairs
			$attributes = array_merge( $attributes,
				\Elementor\Utils::parse_custom_attributes( $url_control['custom_attributes'] ) );
		}

		if ( $attributes ) {
			$this->add_render_attribute( $element, $attributes, $overwrite );
		}

		return $this;
	}


}