<?php


if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}


/**
 * Elementor accordion widget.
 *
 * Elementor widget that displays a collapsible display of content in an
 * accordion style, showing only one item at a time.
 *
 * @since 1.0.0
 */
class BlackdsnCategory extends \Elementor\Widget_Base
{

    use \Dsn\Element\Blackdsn_Widget_Base;


    /**
     * Get widget name.
     *
     * Retrieve accordion widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'dsn_products_category';
    }


    /**
     * Get widget title.
     *
     * Retrieve accordion widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Blackdsn Products Category', 'blackdsn');
    }

    /**
     * Get widget icon.
     *
     * Retrieve accordion widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-product-categories';
    }

    /**
     * Get widget keywords.
     *
     * Retrieve the list of keywords the widget belongs to.
     *
     * @return array Widget keywords.
     * @since 2.1.0
     * @access public
     *
     */
    public function get_keywords()
    {
        return array_merge($this->dsn_keywords(), ['woocommerce', 'shop', 'store', 'product', 'archive']);
    }


    /**
     * Register accordion widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {


        $this->heading();
        $this->style();


    }


    private function heading()
    {
        $control = $this->getControl();

        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__('Header', 'blackdsn'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );


        $control->addSelect('style_category', [
            '' => esc_html__('Default', 'blackdsn'),
            'dsn-style-wc-1' => esc_html__('Card', 'blackdsn')
        ])
            ->setDefault('')
            ->setPrefix_class()
            ->setLabel(esc_html__("Style", 'blackdsn'))
            ->get();

        $control->addNumber('limit', -1, null, 1)
            ->setLabel(esc_html__('Post Per Page', 'blackdsn'))
            ->setDescription(esc_html__('The number of products to display. Defaults to and -1 (display all)  when listing products, and -1 (display all) for categories.', 'blackdsn'))
            ->setDefault(-1)
            ->get();

        $control->addNumberSlider('columns', 1, 9, 1)
            ->setLabel(esc_html__("Columns", 'blackdsn'))
            ->setDefaultRange(4)
            ->setDescription(esc_html__('The number of columns to display. Defaults to 4.', 'blackdsn'))
            ->get();


        $control->addSwitcher('hide_empty')
            ->setLabel(esc_html__('Hide Empty', 'blackdsn'))
            ->setDefault('1')
            ->setLabel_on('Show')
            ->setLabel_off('Hide')
            ->get();


        $control->addSelect('orderby', [
            'slug' => esc_html__('Slug', 'blackdsn'),
            'name' => esc_html__('Name', 'blackdsn'),
            'id' => esc_html__('ID', 'blackdsn'),
            'menu_order' => esc_html__('Menu Order', 'blackdsn'),
            'include' => esc_html__('Include', 'blackdsn'),
            'description' => esc_html__('Description', 'blackdsn'),
            'count' => esc_html__('Count', 'blackdsn'),
        ])
            ->setDefault('name')
            ->setSeparatorBefore()
            ->setLabel(esc_html__("Order By", 'blackdsn'))
            ->setDescription(esc_html__("Sorts the products displayed by the entered option. One or more options can be passed by adding both slugs with a space between them", 'blackdsn'))
            ->get();

        $control->addSelect('order', [
            'ASC' => esc_html__('ASC', 'blackdsn'),
            'DESC' => esc_html__('DESC', 'blackdsn')
        ])
            ->setDefault('ASC')
            ->setLabel(esc_html__("order", 'blackdsn'))
            ->setDescription(esc_html__("States whether the product order is ascending (ASC) or descending (DESC)", 'blackdsn'))
            ->get();


        $category = \DesignGrid\BlackdsnOption::getTermsArray(\DesignGrid\BlackdsnOption::PRODUCT_CATEGORY, 'term_id');

        $control->addSelect2('category', $category)
            ->setMultiple()
            ->setLabel(esc_html__('Categories', 'blackdsn'))
            ->setDefault([])
            ->setLabelBlock()
            ->get();

        $parent_options = ['0' => esc_html__('Only Top Level', 'elementor-pro')] + $category;

        $control->addSelect('parent', $parent_options)
            ->setLabel(esc_html__('Parent', 'blackdsn'))
            ->setDefault('0')
            ->get();


        $this->end_controls_section();

    }


    private function style()
    {
        $control = $this->getControl();

        $this->start_controls_section(
            'style_section',
            [
                'label' => esc_html__('Header', 'blackdsn'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $control->addSlider( 'img_wc_height',
            $control->getDefaultWidthHeight( 'vh' ) )
            ->setLabel( esc_html__( 'Height','blackdsn' ) )
            ->setSelectors( '.woocommerce ul.products li.product a img', 'height:{{SIZE}}{{UNIT}};max-height:{{SIZE}}{{UNIT}};' )
            ->getResponsive();

        $this->end_controls_section();

    }


    /**
     * Render accordion widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        echo blackdsn_shortcode_render_group('products-category', array('widget-base' => $this));
    }

}
