<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}


/**
 * Elementor accordion widget.
 *
 * Elementor widget that displays a collapsible display of content in an
 * accordion style, showing only one item at a time.
 *
 * @since 1.0.0
 */
class GridaStory extends \Dsn\Element\GridaLayout {

	use \Dsn\Element\Arctit_Widget_Base;
	use \Dsn\Element\PostControl;


	/**
	 * Get widget name.
	 *
	 * Retrieve accordion widget name.
	 *
	 * @return string Widget name.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_name() {
		return 'dsn_story';
	}


	/**
	 * Get widget title.
	 *
	 * Retrieve accordion widget title.
	 *
	 * @return string Widget title.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_title() {
		return __( 'Grida Story', 'grida' );

	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve accordion widget icon.
	 *
	 * @return string Widget icon.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_icon() {
		return 'eicon-posts-grid';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @return array Widget keywords.
	 * @since 2.1.0
	 * @access public
	 *
	 */
	public function get_keywords() {
		return array_merge( $this->dsn_keywords(),
			[ 'portfolio', 'posts', 'cpt', 'item', 'loop', 'query', 'cards' ] );
	}


	/**
	 * Register accordion widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _register_controls() {


		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Story Content', 'grida' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);


		$control = $this->getControl();


		$control->addNumber( 'post_per_page', - 1, null, 1 )
		        ->setLabel( esc_html__( 'Post Per Page', 'grida' ) )
		        ->setDescription( esc_html__( 'number of posts to show per page , (-1 or 0) (show all posts) is used.', 'grida' ) )
		        ->setDefault( 6 )
		        ->get();

		$control->addImageSzie()
		        ->getGroup();

		$this->getLayout();


		$control->addHtmlTag( 'dsn_html_tag' )
		        ->setSeparator( "before" )
		        ->get();

		$control->addSize( 'font_size' )
		        ->setDefault( 'title-block' )
		        ->get();

		$this->end_controls_section();

		$this->pagination();
		$this->filterBar();

		$this->styleLayoutTab();


		$this->__style_controller_( $control );

	}

	public function __style_controller_( \ArctitControl $control ) {
		$args = array(
			'post-title'          => esc_html__( 'Title', 'grida' )
		);

		foreach ( $args as $id => $value ):


			$this->start_controls_section(
				'style_content_post_' . sanitize_title( $id ),
				[
					'label' => $value,
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);

			if ( $id !== 'post-title' ) {
				$this->Spacing( 'spacing_content_' . sanitize_title( $id ), '.' . ( $id === 'post-meta a' ? 'post-meta' : $id ), $id === 'post-meta a' ? 'bottom' : 'top' )
				     ->getResponsive();
			}

			if ( $id === 'section_description' ) {
				$control->addSwitcher( 'border_left' )
				        ->setLabel( "Border Left" )
				        ->setReturn_value( 'border-before' )
				        ->get();
			}

			$control->addColor( 'color_content_' . sanitize_title( $id ) )
			        ->setLabel( esc_html__( "Color", 'grida' ) )
			        ->setSeparator( "before" )
			        ->setSelectors( '.' . $id, 'color:{{VALUE}};' )
			        ->get();

			$control->addTypography( 'item_typo_content_' . sanitize_title( $id ), '.' . $id )
			        ->getGroup();

			$control->addTextShadow( 'text_content_shadow_' . sanitize_title( $id ), '.' . $id )
			        ->getGroup();

			$this->end_controls_section();


		endforeach;
	}


	private function pagination() {
		$control = $this->getControl();

		$this->start_controls_section(
			'pagination_section',
			[
				'label' => esc_html__( 'Pagination',
					'grida' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);


		$control->addSelect( 'pagination_type',
			[
				'none' => __( 'None',
					'grida' ),
				'nav'  => __( 'Pagination With Number',
					'grida' ),
				'ajax' => __( 'Button Ajax',
					'grida' ),
			] )
		        ->setLabel( esc_html__( 'Pagination',
			        'grida' ) )
		        ->setDefault( 'none' )
		        ->get();

		$control->addText( 'next_post' )
		        ->setLabel( esc_html__( 'Next Post Title',
			        'grida' ),
			        true )
		        ->setDynamicActive( true )
		        ->setDefault( __( 'NEXT',
			        'grida' ) )
		        ->setConditions( 'pagination_type',
			        'nav' )
		        ->get();

		$control->addText( 'prev_post' )
		        ->setLabel( esc_html__( 'Prev Post Title',
			        'grida' ),
			        true )
		        ->setDynamicActive( true )
		        ->setDefault( __( 'PREV',
			        'grida' ) )
		        ->setConditions( 'pagination_type',
			        'nav' )
		        ->get();


		$control->addText( 'load_more_post' )
		        ->setLabel( esc_html__( 'Load More Title',
			        'grida' ),
			        true )
		        ->setDynamicActive( true )
		        ->setDefault( __( 'Load More',
			        'grida' ) )
		        ->setConditions( 'pagination_type',
			        'ajax' )
		        ->get();


		$control->addText( 'no_more_post' )
		        ->setLabel( esc_html__( 'No More Title',
			        'grida' ),
			        true )
		        ->setDynamicActive( true )
		        ->setDefault( __( 'No More',
			        'grida' ) )
		        ->setConditions( 'pagination_type',
			        'ajax' )
		        ->get();

		$control->getJustifyContent( 'pagination_justify_content' )
		        ->setSeparator( "before" )
		        ->setConditions( 'pagination_type',
			        '!==',
			        'none' )
		        ->get();


		$this->end_controls_section();
	}

	private function filterBar() {
		$control = $this->getControl();

		$this->start_controls_section(
			'filter_bar_section',
			[
				'label' => esc_html__( 'Filter Bar',
					'grida' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);


		$control->addSwitcher( 'filter_bar' )
		        ->setLabel( esc_html__( 'Filter Bar',
			        'grida' ) )
		        ->get();

		$control->addText( 'filter_title' )
		        ->setLabel( esc_html__( 'Filter Title',
			        'grida' ),
			        true )
		        ->setDynamicActive( true )
		        ->setDefault( __( 'Filter',
			        'grida' ) )
		        ->setConditions( 'filter_bar',
			        '1' )
		        ->get();

		$control->addText( 'all_filter' )
		        ->setLabel( esc_html__( 'Button All Filter',
			        'grida' ),
			        true )
		        ->setDynamicActive( true )
		        ->setDefault( __( 'All',
			        'grida' ) )
		        ->setConditions( 'filter_bar',
			        '1' )
		        ->get();

		$control->getJustifyContent( 'filter_bar_justify_content' )
		        ->setSeparator( "before" )
		        ->setConditions( 'filter_bar',
			        '1' )
		        ->get();

		$this->Spacing( 'dsn-filtering-space',
			'.dsn-filtering' )
		     ->setConditions( 'filter_bar',
			     '1' )
		     ->get();

		$this->end_controls_section();
	}


	private function styleLayoutTab() {
		$control = $this->getControl();


		$this->start_controls_section(
			'style_section',
			[
				'label' => esc_html__( 'Layout',
					'grida' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);
		$this->getGridSpace();

		$control->addSlider( 'height',
			$control->getDefaultWidthHeight( 'vh' ) )
		        ->setLabel( esc_html__( 'Height',
			        'grida' ) )
		        ->setSelectors( '.link-pop.has-image', 'height:{{SIZE}}{{UNIT}};max-height:{{SIZE}}{{UNIT}};' )
		        ->getResponsive();

		$this->end_controls_section();


	}


	/**
	 * @param $id
	 * @param $key
	 * @param string $value
	 *
	 * @return ArctitControl|null
	 */
	private function Spacing( $id, $key, string $value = 'bottom' ) {
		$control = $this->getControl();

		return $control->addSlider( $id, [
			'label' => esc_html__( 'Spacing', 'grida' ),
			'range' => [
				'px' => [ 'max' => 100, ],
			],
		] )->setSelectors( $key, 'margin-' . $value . ':{{SIZE}}{{UNIT}}' );

	}


	/**
	 * Render accordion widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {

		$shortcode     = new arctitShortCode( array( 'widget-base' => $this ) );
		$post_per_page = $shortcode->getVal( 'post_per_page', 6 );
		$CurrentPage   = ( get_query_var( 'paged' ) ) ? get_query_var( 'paged' ) : 1;


		$option_query = array(
			'paged'          => $CurrentPage,
			'posts_per_page' => $post_per_page,
			'post_type'      => grida_story_slug(),
		);

		$myposts = new WP_Query( $option_query );


		$this->addPrefixClass( 'dsn-stories', $shortcode );
		$this->addAnimateFade( 'dsn-item-story', $shortcode );
		$this->add_render_attribute( 'dsn-stories',
			[
				'class' => [
					'dsn-stories dsn-posts dsn-post-type-' . $shortcode->getVal( 'style_post', 'classic' ),
					$shortcode->getVal( 'style_cards_post', '' )
				]
			] );


		?>
        <div class="root-posts"
             style="--dsn-row-item:<?php echo esc_attr( $shortcode->getValueNumberSlide( 'row_layout_gap', 30 ) ) ?>px">
			<?php

			if ( $shortcode->getVal( 'filter_bar' ) ):
				$this->render_filter_bar( $shortcode, grida_get_filter_categories( $myposts, grida_category_story_slug() ) );
			endif;

			printf( '<div %s>%s</div>',
				$this->get_render_attribute_string( 'dsn-stories' ),
				grida_shortcode_render_group( 'story', array( 'widget-base' => $this ), $myposts )
			);


			$type_nav_post = $shortcode->getVal( 'pagination_type', 'none' );


			if ( $type_nav_post !== 'none' && $myposts->max_num_pages > 1 ):

				echo '<div class="dsn-p-pag dsn-stories-el d-flex type-p-' . esc_attr( $type_nav_post ) . ' ' . esc_attr( $shortcode->getVal( 'pagination_justify_content', 'justify-content-center' ) ) . ' ">';
				if ( $myposts->have_posts() && $type_nav_post === 'ajax' ) :
					$options = array(
						'image_size'            => 'large',

					);

					$attr_query = array();
					foreach ( $options as $key => $val ):
						if ( $key === 'excerpt_length' ) {
							$attr_query[ $key ] = $shortcode->getValueNumberSlide( $key,
								$val );
						} else {
							$attr_query[ $key ] = $shortcode->getVal( $key,
								$val );
						}
					endforeach;

                    $option_query['action'] = 'grida_story_query';

					$this->add_render_attribute( 'ajax_attr',
						[
							'class'           => [
								'button-load-more p-relative d-block dsn-btn border-radius move-circle image-zoom background-section',
								esc_attr( $shortcode->getVal( 'style_space_pag' ) )
							],
							'data-dsn'        => 'parallax',
							'data-dsn-option' => json_encode( $option_query ),
							'data-dsn-attr'   => json_encode( $attr_query ),
						] );
					?>
                    <div <?php $this->print_render_attribute_string( 'ajax_attr' ) ?>>
                        <span class="dsn-load-progress-ajax h-100">01</span>
                        <span class="progress-text progress-load-more d-block"><?php echo $shortcode->getVal( 'load_more_post', __( 'Load More', 'grida' ) ) ?></span>
                        <span class="progress-text progress-no-more d-block"><?php echo $shortcode->getVal( 'no_more_post', __( 'No More', 'grida' ) ) ?></span>
                    </div>
				<?php endif;
				if ( $type_nav_post === 'nav' ) :
					\DesignGrid\gridaOption::Pagination( array(
						'prev_text'          => grida_buttons_pagination( $shortcode->getVal( 'prev_post', __( 'PREV', 'grida' ) ), 'dsn-prev' ),
						'next_text'          => grida_buttons_pagination( $shortcode->getVal( 'next_post', __( 'NEXT', 'grida' ) ) ),
						'before_page_number' => '<span class="dsn-numbers dsn-heading-title title-tag"><span class="number">',
						'after_page_number'  => '</span></span>',
						'total'              => $myposts->max_num_pages,

					), esc_attr( $shortcode->getVal( 'style_space_pag' ) ) );

				endif;
				echo '</div>';
			endif;

			?>
        </div>

		<?php


	}


	private function render_filter_bar( arctitShortCode $shortcode, $terms ) {


		if ( ! empty( $terms ) ):
			$this->add_inline_editing_attributes( 'filter_title', 'none' );
			$this->add_inline_editing_attributes( 'all_filter', 'none' );

			$this->add_render_attribute( 'filter_title', 'class', 'filter-title title-block border-before mr-20' );
			$this->add_render_attribute( 'all_filter', [
				'class'       => 'active',
				'data-filter' => '*',
				'type'        => 'button',
			] );

			?>
            <div class="dsn-filtering p-relative w-100 mb-50 <?php echo esc_attr( $shortcode->getVal( 'filter_bar_justify_content', 'justify-content-center' ) ) ?>">
                <div class="filtering-t">
                    <div class="filtering-wrap d-flex justify-content-between">
                        <h5 <?php $this->print_render_attribute_string( 'filter_title' ) ?> >
							<?php echo esc_html( $shortcode->getVal( 'filter_title', esc_html__( 'Filter', 'grida' ) ) ) ?>
                        </h5>
                        <div class="filtering">
                            <button <?php $this->print_render_attribute_string( 'all_filter' ) ?> >
								<?php echo esc_html( $shortcode->getVal( 'all_filter', esc_html__( 'All', 'grida' ) ) ) ?>
                            </button>
							<?php
							foreach ( $terms as $slug => $name ):?>
                                <button type="button"
                                        class="<?php echo 'dsn-filter-category-' . esc_attr( $slug ); ?>"
                                        data-filter='.<?php echo esc_attr( $slug ); ?> , .<?php echo 'category-' . esc_attr( $slug ); ?>'>
									<?php echo esc_html( $name ); ?>
                                </button>
							<?php endforeach;
							?>
                        </div>
                    </div>
                </div>
            </div>


		<?php endif;
	}


}
