<?php

use Elementor\Controls_Manager;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Css_Filter;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Typography;
use Elementor\Repeater;
use Elementor\Utils;

class BlackdsnControl {

	private $element_base = null;
	private $element = null;
	private $isGroup = false;

	private $id = null;
	private $args = [];


	public function __construct( $element_base ) {
		$this->element_base = $element_base;
		$this->element      = $element_base;
	}


	private function getOptionArray( $options, $id, $default = false ) {
		return blackdsn_get_option_array( $options, $id, $default );
	}

	public function setLabel( $label, $placeholder = false ): BlackdsnControl {
		$this->args['label'] = $label;
		if ( $placeholder ) {
			$this->setPlaceholder( $label );
		}


		return $this;
	}


	public function setToggle( $label = true ): BlackdsnControl {
		$this->args['toggle'] = $label;

		return $this;
	}

	/**
	 * @return null
	 */
	public function get_element_base() {
		return $this->element_base;
	}


	/**
	 * @param bool $label
	 *
	 * @return $this
	 */
	public function setLabelBlock( bool $label = true ): BlackdsnControl {
		$this->args['label_block'] = $label;

		return $this;
	}

	public function setPlaceholder( $placeholder ): BlackdsnControl {
		$this->args['placeholder'] = $placeholder;

		return $this;
	}

	/**
	 * @param $default
	 *
	 * @return $this
	 */
	public function setDefault( $default ): BlackdsnControl {
		$this->args['default'] = $default;

		return $this;
	}


	public function setDescription( $default ): BlackdsnControl {
		$this->args['description'] = $default;

		return $this;
	}


	public function setSeparator( $separator ): BlackdsnControl {
		$this->args['separator'] = $separator;

		return $this;
	}

	public function setSeparatorAfter(): BlackdsnControl {
		return $this->setSeparator( "after" );
	}

	public function setSeparatorBefore(): BlackdsnControl {
		return $this->setSeparator( "before" );
	}

	/**
	 * @param string $default
	 *
	 * @return $this
	 */
	public function setDefault_url( string $default = '' ): BlackdsnControl {
		$this->args['default']['url'] = $default;

		return $this;
	}

	/**
	 * @param bool|string $default
	 *
	 * @return $this
	 */
	public function setDefault_is_external( $default = false ): BlackdsnControl {
		$this->args['default']['is_external'] = $default;

		return $this;
	}

	/**
	 * @param false|string $default
	 *
	 * @return $this
	 */
	public function setDefault_is_nofollow( $default = false ): BlackdsnControl {
		$this->args['default']['nofollow'] = $default;

		return $this;
	}


	public function setDynamicActive( $default ): BlackdsnControl {

		if ( ! isset( $this->args['dynamic']['active'] ) ) {
			$this->args['dynamic']['active'] = $default;
		}

		return $this;
	}


	public function setLabel_on( $label_on ): BlackdsnControl {
		$this->args['label_on'] = $label_on;

		return $this;
	}

	public function setLabel_off( $label_off ): BlackdsnControl {
		$this->args['label_off'] = $label_off;

		return $this;
	}


	public function setReturn_value( $return_value ): BlackdsnControl {
		$this->args['return_value'] = (string) $return_value;

		return $this;
	}

	public function setTitle_field( $return_value ): BlackdsnControl {
		$this->args['title_field'] = "{{{ $return_value }}}";

		return $this;
	}

	public function setTitle_field_withIcon( $icon, $return_value ): BlackdsnControl {
		$this->args['title_field'] = '{{{ elementor.helpers.renderIcon( this, ' . $icon . ', {}, "i", "panel" ) || \'<i class="{{ ' . $icon . ' }}" aria-hidden="true"></i>\' }}} {{{ ' . $return_value . ' }}}';

		return $this;
	}

	public function setTitle_field_WithImage( $image, $return_value ): BlackdsnControl {
		$this->args['title_field'] = '<img src="{{{ elementor.imagesManager.getImageUrl( ' . $image . ' ) }}}" width="50" style="border-radius: 5px;max-height: 30px;object-fit: cover;vertical-align: middle;margin-right: 5px;" /> {{{ ' . $return_value . ' }}}';

		return $this;
	}

	public function setRows( $rows ): BlackdsnControl {
		$this->args['rows'] = $rows;

		return $this;
	}


	/**
	 * @param array $condition
	 *
	 * @return $this
	 */
	public function setCondition( array $condition ): BlackdsnControl {
		$this->args['condition'] = $condition;

		return $this;
	}

	/**
	 * @param string $relation
	 *
	 * @return $this
	 */
	public function setRelation( string $relation = "or" ): BlackdsnControl {
		$this->args['conditions'] ['relation'] = $relation;

		return $this;
	}


	/**
	 * @param $min
	 * @param $max
	 *
	 * @return $this
	 */
	public function setRangePx( $min, $max ): BlackdsnControl {
		$this->args['range'] ['px'] = [
			'min' => $min,
			'max' => $max,
		];

		return $this;
	}

	/**
	 * @param $default
	 *
	 * @return $this
	 */
	public function setDefaultDesktop( $default ): BlackdsnControl {
		$this->args['desktop_default'] = $default;

		return $this;
	}

	/**
	 * @param $size
	 * @param string $unit
	 *
	 * @return $this
	 */
	public function setDefaultRange( $size, string $unit = 'px' ): BlackdsnControl {
		$this->args['default'] = [
			'size' => $size,
			'unit' => $unit,
		];

		return $this;
	}


	public function setDefaultDesktopRange( $size, $unit = 'px' ): BlackdsnControl {
		$this->setDefaultDesktop( [
			'size' => $size,
			'unit' => $unit,
		] );

		return $this;
	}

	/**
	 * @param $default
	 *
	 * @return $this
	 */
	public function setDefaultTablet( $default ): BlackdsnControl {
		$this->args['tablet_default'] = $default;


		return $this;
	}

	/**
	 * @param $size
	 * @param string $unit
	 *
	 * @return $this
	 */
	public function setDefaultTabletRange( $size, string $unit = 'px' ): BlackdsnControl {
		$this->setDefaultTablet( [
			'size' => $size,
			'unit' => $unit,
		] );

		return $this;
	}

	/**
	 * @param $default
	 *
	 * @return $this
	 */
	public function setDefaultMobile( $default ): BlackdsnControl {
		$this->args['mobile_default'] = $default;

		return $this;
	}

	/**
	 * @param $size
	 * @param string $unit
	 *
	 * @return $this
	 */
	public function setDefaultMobileRange( $size, string $unit = 'px' ): BlackdsnControl {
		$this->setDefaultMobile( [
			'size' => $size,
			'unit' => $unit,
		] );

		return $this;
	}

	/**
	 * @param $key
	 * @param $value
	 *
	 * @return $this
	 */
	public function setSelectors( $key, $value ): BlackdsnControl {
		$this->args['selectors'] [ '{{WRAPPER}} ' . $key ] = $value;

		return $this;

	}

	/**
	 * @param bool $value
	 *
	 * @return $this
	 */
	public function setMultiple( bool $value = true ): BlackdsnControl {
		$this->args['multiple'] = $value;

		return $this;
	}

	/**
	 * @param $key
	 * @param $value
	 *
	 * @return $this
	 */
	public function setSelectorRepeater( $key, $value ): BlackdsnControl {
		$this->args['selectors'] [ '{{WRAPPER}} {{CURRENT_ITEM}} ' . $key ] = $value;

		return $this;

	}

	/**
	 * @param $key
	 *
	 * @return $this;
	 */
	public function setSelectorGrid( $key ): BlackdsnControl {
		$this->setSelectors( $key, 'grid-template-columns: repeat({{SIZE}},1fr);' );

		return $this;
	}


	public function setPrefix_class( $prefix_class = '' ): BlackdsnControl {

		$this->args['prefix_class'] = $prefix_class;

		return $this;
	}


	/**
	 * @param $key
	 * @param null $operation
	 * @param null $value
	 * @param string $relation
	 *
	 * One of and/or. If omitted, defaults to and.
	 * terms    An array of arrays containing the rules.
	 * name    The parent control’s name.
	 *
	 * Access responsive variations by suffixing with _tablet or _mobile.
	 * Important! Access values in complex Unit/Multiple controls with:
	 * scale_mobile[size] or link[url]
	 * (for Slider or URL, respectively.)
	 *
	 * operator    Defaults to strict equality (===), when omitted. Otherwise, use one of:
	 *
	 * ==
	 * !=
	 * !==
	 * in
	 * !in
	 * contains
	 * !contains
	 * <
	 * <=
	 * >
	 * >=
	 * ===
	 *
	 * value
	 *
	 * Some option’s value from the parent control.
	 * It needs to be an array when using in and !in operators.
	 *
	 * @return $this
	 */
	public function setConditions( $key, $operation = null, $value = null, string $relation = 'or' ): BlackdsnControl {

		if ( is_array( $key ) ) {
			$this->args['conditions'] ['relation'] = $relation;
			$this->args['conditions'] ['terms']    = $key;

			return $this;
		}


		if ( $value === null ) {
			$value     = $operation;
			$operation = '===';
		}


		$this->args['conditions'] ['relation'] = $relation;
		$this->args['conditions'] ['terms'][]  = [
			'name'     => $key,
			'operator' => $operation,
			'value'    => $value,
		];

		return $this;
	}


	/**
	 * @param $id
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addText( $id, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::TEXT;
		$this->args   = $args;

		return $this;
	}

	/**
	 * @param $id
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addHidden( $id, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::HIDDEN;
		$this->args   = $args;

		return $this;
	}


	public function addHiddenNoSpace( $id, array $args = array() ): BlackdsnControl {

		$this->addHidden( $id, $args )
		     ->setDefault( 'dsn-mb-no-space' )
		     ->setPrefix_class();

		return $this;
	}


	public function addPopover( $id, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::POPOVER_TOGGLE;
		$this->args   = $args;

		return $this;
	}

	/**
	 * @param string $id
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addImage( string $id = 'image', array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::MEDIA;
		$this->args   = array_merge( [
			'label'   => __( 'Choose Image', 'elementor' ),
			'dynamic' => [
				'active' => true,
			],
			'default' => [
				'url' => Utils::get_placeholder_image_src(),
			],
		], $args );

		return $this;
	}


	public function addGallery( $id = 'gallery', array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::GALLERY;
		$this->args   = array_merge( [
			'label'   => __( 'Galleries', 'blackdsn' ),
			'dynamic' => [
				'active' => true,
			],
		], $args );

		return $this;
	}


	public function getGroupImage( $id = 'image', array $args = array() ): BlackdsnControl {
		$this->addImage( $id, $this->getOptionArray( $args, 'image', array() ) )
		     ->get()
		     ->addImageSize( $id, $this->getOptionArray( $args, 'size', array() ) )
		     ->getGroup()
		     ->getCaption();

		return $this;
	}


	public function getGroupGallery( $id = 'gallery', array $args = array() ): BlackdsnControl {
		$this->addGallery( $id, $this->getOptionArray( $args, 'gallery', array() ) )
		     ->get()
		     ->addImageSize( $id, $this->getOptionArray( $args, 'size', array() ) )
		     ->getGroup()
		     ->getCaption();

		return $this;
	}

	public function getCaption( $id = 'caption_source', $id_custom_caption = 'caption' ): BlackdsnControl {


		$this->addSelect( $id, [
			'none'       => __( 'None', 'elementor' ),
			'attachment' => __( 'Attachment Caption', 'elementor' ),
			'custom'     => __( 'Custom Caption', 'elementor' ),
		],
			[
				'label'   => __( 'Caption', 'elementor' ),
				'default' => 'none',
			] )
		     ->get()
		     ->addText( $id_custom_caption, [
			     'label'       => __( 'Custom Caption', 'elementor' ),
			     'default'     => __( 'Enter your image caption', 'elementor' ),
			     'placeholder' => __( 'Enter your image caption', 'elementor' ),
			     'condition'   => [
				     'caption_source' => 'custom',
			     ],
			     'dynamic'     => [
				     'active' => true,
			     ],
		     ] )->get();


		return $this;
	}

	/**
	 * @param $id
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addDivider( $id, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::DIVIDER;
		$this->args   = $args;

		return $this;
	}


	/**
	 * @param $id
	 * @param array $options
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addSelect( $id, array $options = array(), array $args = array() ): BlackdsnControl {

		$this->id        = $id;
		$args['type']    = Controls_Manager::SELECT;
		$args['options'] = $options;
		$this->args      = $args;

		return $this;
	}

	/**
	 * @param $id
	 * @param array $options
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addSelectDsn( $id, array $options = array(), array $args = array() ): BlackdsnControl {

		$this->id        = $id;
		$args['type']    = "dsn_select";
		$args['options'] = $options;
		$this->args      = $args;

		return $this;
	}


	/**
	 * @param $id
	 * @param array $options
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addSelect2( $id, array $options = array(), array $args = array() ): BlackdsnControl {

		$this->id        = $id;
		$args['type']    = Controls_Manager::SELECT2;
		$args['options'] = $options;
		$this->args      = $args;

		return $this;
	}


	public function addSlider( $id, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::SLIDER;
		$this->args   = $args;

		return $this;
	}

	/**
	 * @param string $vwh
	 *
	 * @return array
	 */
	public function getDefaultWidthHeight( string $vwh = 'vw' ): array {
		return [
			'default'        => [
				'unit' => '%',
			],
			'tablet_default' => [
				'unit' => '%',
			],
			'mobile_default' => [
				'unit' => '%',
			],
			'size_units'     => [ '%', 'px', $vwh ],
			'range'          => [
				'%'  => [
					'min' => 1,
					'max' => 100,
				],
				'px' => [
					'min' => 1,
					'max' => 1000,
				],
				$vwh => [
					'min' => 1,
					'max' => 100,
				],
			],
		];
	}


	/**
	 * @return array
	 */
	public function getStyleButton(): array {
		return array(
			'dsn-normal-btn'             => esc_html__( 'Default', 'blackdsn' ),
			'dsn-def-btn'                => esc_html__( 'Blackdsn Button', 'blackdsn' ),
			'dsn-def-btn dsn-hover-icon' => esc_html__( 'Blackdsn Hover', 'blackdsn' )
		);
	}

	/**
	 * @return array
	 */
	public function getBorderColor(): array {
		return array(
			'border-color-default'     => esc_html__( 'Default', 'blackdsn' ),
			'border-color-theme-color' => esc_html__( 'Theme Color', 'blackdsn' ),
			'border-color-main'        => esc_html__( 'Main Color', 'blackdsn' ),
			'border-color-assistant'   => esc_html__( 'Assistant Color', 'blackdsn' ),
			'border-color-heading'     => esc_html__( 'Heading Color', 'blackdsn' ),
			'border-color-body'        => esc_html__( 'Body Color', 'blackdsn' ),
		);
	}

	/**
	 * @param string $id
	 * @param string $key
	 * @param int|float $desktop
	 * @param int|float $table
	 * @param int|float $mobile
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addGrid(
		string $id = 'col_grid',
		string $key = '.d-grid.dsn-effect-grid',
		$desktop = 3,
		$table = 2,
		$mobile = 1,
		array $args = array()
	): BlackdsnControl {
		$this->setRangePx( 1, 6 );
		$this->addSlider( $id, $args );
		$this->setDefaultDesktopRange( $desktop );
		$this->setDefaultTabletRange( $table );
		$this->setDefaultMobileRange( $mobile );
		$this->setSelectorGrid( $key );
		$this->setLabel( esc_html__( 'Columns', 'blackdsn' ) );

		return $this;
	}


	/**
	 * @param string $id
	 * @param string $key
	 * @param int|float $desktop
	 *
	 * @return $this
	 */
	public function addGridGapColumns(
		string $id = 'dsn_col_gap',
		string $key = '.d-grid.dsn-effect-grid',
		$desktop = 30
	): BlackdsnControl {

		$this->addSlider( $id, [
			'label'           => esc_html__( 'Columns Gap', 'blackdsn' ),
			'desktop_default' => [
				'size' => $desktop,
				'unit' => 'px',
			],
			'range'           => [
				'px' => [
					'min' => 0,
					'max' => 100,
				],
			],
			'selectors'       => [
				'{{WRAPPER}} ' . $key => 'grid-column-gap: {{SIZE}}px;',
			],
		] );

		return $this;
	}

	/**
	 * @param string $id
	 * @param string $key
	 * @param int|float $desktop
	 * @param int|float $tablet
	 *
	 * @return $this
	 */
	public function addGridGapRow(
		string $id = 'dsn_row_gap',
		string $key = '.d-grid.dsn-effect-grid',
		$desktop = 50,
		$tablet = 30
	): BlackdsnControl {

		$this->addSlider( $id, [
			'label'           => esc_html__( 'Rows Gap', 'blackdsn' ),
			'desktop_default' => [
				'size' => $desktop,
				'unit' => 'px',
			],
			'tablet_default'  => [
				'size' => $tablet,
				'unit' => 'px',
			],
			'range'           => [
				'px' => [
					'min' => 0,
					'max' => 100,
				],
			],
			'selectors'       => [
				'{{WRAPPER}} ' . $key => 'grid-row-gap: {{SIZE}}px;',
			],
		] );

		return $this;

	}

	/**
	 * @param $id
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addTextarea( $id, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::TEXTAREA;
		$this->args   = $args;

		return $this;
	}

	/**
	 * @param $id
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addTextareaEditor( $id, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::WYSIWYG;
		$this->args   = $args;

		return $this;
	}


	/**
	 * @param $id
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addLink( $id, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::URL;
		$this->args   = $args;

		return $this;
	}


	/**
	 * @param $id
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addSwitcher( $id, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::SWITCHER;
		$this->args   = $args;
		$this->setReturn_value( "1" );

		return $this;
	}

	public function addNumber( $id, $min, $max, $step, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::NUMBER;
		$this->args   = array_merge( $args, [ 'min' => $min, 'max' => $max, 'step' => $step ] );

		return $this;
	}

	public function addNumberSlider( $id, $min, $max, $step, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::SLIDER;
		$this->args   = array_merge( $args, [
				'range' => [
					'px' => [ 'min' => $min, 'max' => $max, 'step' => $step ],
				],
			]
		);


		return $this;
	}

	/**
	 * @param string[] $value
	 *
	 * @return $this
	 */
	public function setSizeUnits( array $value = [ 'px' ] ): BlackdsnControl {
		$this->args['size_units'] = $value;

		return $this;
	}


	/**
	 * @param $id
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addChoose( $id, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::CHOOSE;

		$this->args = $args;

		return $this;
	}

	public function setOptionChoose( $key, $title, $icon ): BlackdsnControl {
		$this->args['options'][ $key ] = [ 'title' => $title, 'icon' => $icon ];

		return $this;
	}


	public function addSize( $id = 'font_size', $options = array(), array $args = array() ): BlackdsnControl {

		return $this->addSelect( $id, array_merge( [
			''               => esc_html__( 'Default', 'blackdsn' ),
			'title'          => esc_html__( 'Title Section Large', 'blackdsn' ),
			'title-h2'       => esc_html__( 'Title Section Default', 'blackdsn' ),
			'title-block'    => esc_html__( 'Title Block', 'blackdsn' ),
			'sm-title-block' => esc_html__( 'Title Block Small ', 'blackdsn' ),
			'sub-heading'    => esc_html__( 'Paragraph Large ', 'blackdsn' ),
			'sm-p'           => esc_html__( 'Paragraph Small ', 'blackdsn' ),
			'font-number'    => esc_html__( 'Number', 'blackdsn' ),

		], $options ), $args )
		            ->setDefault( '' )
		            ->setLabel( __( 'Size', 'blackdsn' ) );
	}


	/**
	 * @param $id
	 * @param $key
	 * @param string $value
	 *
	 * @return BlackdsnControl
	 */
	public function addSpacing( $id, $key, string $value = 'bottom' ): BlackdsnControl {


		return $this->addSlider( $id, [
			'label' => esc_html__( 'Spacing', 'blackdsn' ),
			'range' => [
				'px' => [ 'max' => 100, ],
			],
		] )->setSelectors( $key, 'margin-' . $value . ':{{SIZE}}{{UNIT}}' );

	}


	/**
	 * @param string $id
	 * @param array $options
	 * @param array $args
	 *
	 * @return BlackdsnControl
	 */
	public function addLineText(
		string $id = 'dsn_line_text',
		array $options = array(),
		array $args = array()
	): BlackdsnControl {

		return $this->addSelect( $id, array_merge( [
			''                           => esc_html__( 'Default', 'blackdsn' ),
			'border-section-bottom'      => esc_html__( 'Border Bottom', 'blackdsn' ),
			'p-10 background-main'       => esc_html__( 'Background Main', 'blackdsn' ),
			'p-10 background-section'    => esc_html__( 'Background Section', 'blackdsn' ),
			'p-10 background-theme'      => esc_html__( 'Background Theme', 'blackdsn' ),
			'square-after'               => esc_html__( 'Square After', 'blackdsn' ),
			'square-before'              => esc_html__( 'Square Before', 'blackdsn' ),
			'square-before square-after' => esc_html__( 'Both Square (After , Before )', 'blackdsn' ),
			'circle-after'               => esc_html__( 'Circle After', 'blackdsn' ),
			'circle-before'              => esc_html__( 'Circle Before', 'blackdsn' ),
			'circle-before circle-after' => esc_html__( 'Both Circle (After , Before )', 'blackdsn' ),
		], $options ), $args )
		            ->setDefault( '' )
		            ->setLabel( __( 'Line Text', 'blackdsn' ) );
	}

	public function addHtmlTag( $id = 'dsn_html_tag', $options = array(), array $args = array() ): BlackdsnControl {

		return $this->addSelect( $id, array_merge( [
			'h1'   => 'H1',
			'h2'   => 'H2',
			'h3'   => 'H3',
			'h4'   => 'H4',
			'h5'   => 'H5',
			'h6'   => 'H6',
			'div'  => 'div',
			'span' => 'span',
			'p'    => 'p',

		], $options ), $args )->setDefault( 'h2' )->setLabel( __( 'HTML Tag', 'blackdsn' ) );
	}


	public function startRepeater(): BlackdsnControl {
		$this->element_base = new Repeater();

		return $this;
	}

	public function endRepeater( $id, array $args = array() ): BlackdsnControl {
		$repeater           = $this->element_base;
		$this->element_base = $this->element;
		$this->id           = $id;
		$args['type']       = Controls_Manager::REPEATER;
		$args['fields']     = $repeater->get_controls();
		$this->args         = $args;


		return $this;
	}


	/**
	 * @param $id
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addColor( $id, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::COLOR;
		$this->args   = $args;

		return $this;
	}


	/**
	 * @param $id
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addIcons( $id, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::ICONS;
		$this->args   = $args;

		return $this;
	}

	/**
	 * @param $id
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addIcon( $id, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::ICON;
		$this->args   = $args;

		return $this;
	}

	public function addHeadingColor( $id, $selector, array $args = array() ): BlackdsnControl {
		$this->addSelect( $id, [
			''              => __( 'Default', 'blackdsn' ),
			'body-color'    => __( 'Body Color', 'blackdsn' ),
			'heading-color' => __( 'Heading Color', 'blackdsn' ),
			'theme-color'   => __( 'Theme Color', 'blackdsn' ),
			'custom'        => __( 'Custom Color', 'blackdsn' ),
		], array_merge( [ 'default' => '', 'label' => __( 'Color', 'blackdsn' ) ], $args ) )->get();

		$this->id     = $id . '_group';
		$args['type'] = Controls_Manager::COLOR;
		$this->args   = $args;
		$this->setConditions( $id, '===', 'custom' );
		if ( is_array( $selector ) ) {
			foreach ( $selector as $select ):
				$this->setSelectors( $select, 'color: {{VALUE}};' );
			endforeach;
		} else {
			$this->setSelectors( $selector, 'color: {{VALUE}};' );
		}

		return $this;
	}


	/**
	 * @param $id
	 * @param string $selector
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addTypography(
		$id,
		string $selector = '.dsn-heading-title',
		array $args = array()
	): BlackdsnControl {

		$this->id     = Group_Control_Typography::get_type();
		$args['name'] = $id;
		$this->args   = array_merge( [ 'selector' => '{{WRAPPER}} ' . $selector ], $args );

		$this->isGroup = true;


		return $this;
	}

	/**
	 * @param string $id
	 * @param string $selector
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addPaddingGroup(
		string $id = 'item_padding',
		string $selector = '.dsn-heading-title',
		array $args = array()
	): BlackdsnControl {


		$this->id     = $id;
		$args['type'] = Controls_Manager::DIMENSIONS;
		$this->args   = array_merge( $args, [
				'label'      => __( 'Padding', 'elementor' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} ' . $selector => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);


		return $this;
	}

	/**
	 * @param string $id
	 * @param string $selector
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addBorderRadiusGroup(
		string $id = 'item_border_radius',
		string $selector = '.dsn-heading-title',
		array $args = array()
	): BlackdsnControl {


		$this->id     = $id;
		$args['type'] = Controls_Manager::DIMENSIONS;
		$this->args   = array_merge( $args, [
				'label'      => __( 'Border Radius', 'blackdsn' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} ' . $selector => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);


		return $this;
	}


	/**
	 * @param string $id
	 * @param string $selector
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addMarginGroup(
		string $id = 'item_margin',
		string $selector = '.dsn-heading-title',
		array $args = array()
	): BlackdsnControl {


		$this->id     = $id;
		$args['type'] = Controls_Manager::DIMENSIONS;
		$this->args   = array_merge( $args, [
				'label'      => __( 'Margin', 'elementor' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} ' . $selector => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);


		return $this;
	}


	/**
	 * @param $id
	 * @param string $selector
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addTextShadow(
		$id,
		string $selector = '.dsn-heading-title',
		array $args = array()
	): BlackdsnControl {

		$this->id     = Group_Control_Text_Shadow::get_type();
		$args['name'] = $id;
		$this->args   = array_merge( [ 'selector' => '{{WRAPPER}} ' . $selector ], $args );

		$this->isGroup = true;


		return $this;
	}


	/**
	 * @param string $id
	 * @param string $selector
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addBoxShadow(
		string $id = 'item_box_shadow',
		string $selector = '.dsn-heading-title',
		array $args = array()
	): BlackdsnControl {

		$this->id     = Group_Control_Box_Shadow::get_type();
		$args['name'] = $id;
		$this->args   = array_merge( [ 'selector' => '{{WRAPPER}} ' . $selector ], $args );

		$this->isGroup = true;


		return $this;
	}

	/**
	 * @param string $id
	 * @param string $selector
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addBorder(
		string $id = 'item_border_style',
		string $selector = '.dsn-heading-title',
		array $args = array()
	): BlackdsnControl {

		$this->id     = Group_Control_Border::get_type();
		$args['name'] = $id;
		$this->args   = array_merge( [ 'selector' => '{{WRAPPER}} ' . $selector ], $args );

		$this->isGroup = true;


		return $this;
	}


	public function addImageSize( $id = 'image', array $args = array() ): BlackdsnControl {

		$this->id     = Group_Control_Image_Size::get_type();
		$args['name'] = $id;
		$this->args   = array_merge( [ 'default' => 'large' ], $args );

		$this->isGroup = true;


		return $this;
	}

	/**
	 * @param string $id
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addFilterImage( string $id = 'css_filters', array $args = array() ): BlackdsnControl {

		$this->id     = Group_Control_Css_Filter::get_type();
		$args['name'] = $id;
		$this->args   = $args;

		$this->isGroup = true;


		return $this;
	}


	/**
	 * @param $id
	 * @param string $selector
	 * @param array $options
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addBlendMode(
		$id,
		string $selector = '.dsn-heading-title',
		array $options = array(),
		array $args = array()
	): BlackdsnControl {


		$this->addSelect( $id, array_merge( [
			''            => esc_html__( 'Normal', 'blackdsn' ),
			'multiply'    => 'Multiply',
			'screen'      => 'Screen',
			'overlay'     => 'Overlay',
			'darken'      => 'Darken',
			'lighten'     => 'Lighten',
			'color-dodge' => 'Color Dodge',
			'saturation'  => 'Saturation',
			'color'       => 'Color',
			'difference'  => 'Difference',
			'exclusion'   => 'Exclusion',
			'hue'         => 'Hue',
			'luminosity'  => 'Luminosity',

		], $options ), $args );

		if ( $selector ) {
			$this->setSelectors( $selector, 'mix-blend-mode: {{VALUE}}' );
		}

		$this->setDefault( '' )
		     ->setLabel( esc_html__( 'Blend Mode', 'blackdsn' ) );


		return $this;
	}


	/**
	 * @param $id
	 * @param float $default
	 * @param array $args
	 *
	 * @return $this
	 */
	public function getTriggerHock( $id, $default = "bottom", array $args = array() ): BlackdsnControl {

		$args = array_merge( [
			'label'       => __( 'Start', 'blackdsn' ),
			'description' => __( "The ScrollTrigger's starting scroll position (numeric, in pixels). This value gets calculated when the ScrollTrigger is refreshed, so anytime the window/scroller gets resized it'll be recalculated",
				'blackdsn' ),
			'default'     => $default,
		], $args );


		return $this->addSelect( $id, [
			'bottom' => esc_html__( "start Section Bottom Window", 'blackdsn' ),
			'center' => esc_html__( "center Window", 'blackdsn' ),
			'top'    => esc_html__( "start Section Top Window", 'blackdsn' ),
		], $args )
		            ->setLabelBlock()
		            ->setDefault( $default );


	}


	/**
	 * @param string $id
	 *
	 * @return $this
	 */
	public function getAlign( string $id = 'align' ): BlackdsnControl {

		$this->addChoose( $id, array(
			'label'        => __( 'Alignment', 'elementor' ),
			'options'      => [
				'left'    => [
					'title' => __( 'Left', 'elementor' ),
					'icon'  => 'eicon-text-align-left',
				],
				'center'  => [
					'title' => __( 'Center', 'elementor' ),
					'icon'  => 'eicon-text-align-center',
				],
				'right'   => [
					'title' => __( 'Right', 'elementor' ),
					'icon'  => 'eicon-text-align-right',
				],
				'justify' => [
					'title' => __( 'Justified', 'elementor' ),
					'icon'  => 'eicon-text-align-justify',
				],
			],
			'prefix_class' => 'elementor%s-align-',
			'default'      => '',
		) );

		return $this;
	}

	/**
	 * @param string $id
	 * @param bool|string $key
	 *
	 * @return $this
	 */

	public function getJustifyContent( string $id = 'justify_content', $key = false ): BlackdsnControl {


		$this->addChoose( $id, array(
			'label'   => __( 'Justify Content', 'elementor' ),
			'options' => [
				'flex-start' => [
					'title' => __( 'Left', 'elementor' ),
					'icon'  => 'eicon-h-align-left',
				],
				'center'     => [
					'title' => __( 'Center', 'elementor' ),
					'icon'  => 'eicon-h-align-center',
				],

				'end'           => [
					'title' => __( 'Right', 'elementor' ),
					'icon'  => 'eicon-h-align-right',
				],
				'space-between' => [
					'title' => __( 'Between', 'blackdsn' ),
					'icon'  => 'eicon-code-bold',
				],
			],

		) );

		if ( $key ) {
			$this->getSelectorjustifyContent( $key );
		}


		return $this;
	}


	/**
	 * @param $key
	 * @param string $value
	 *
	 * @return $this
	 */
	public function getSelectorJustifyContent( $key, string $value = "" ): BlackdsnControl {
		return $this->setSelectors( $key, 'justify-content:{{VALUE}};' . $value );

	}


	/**
	 * @param string $id
	 * @param bool|string $key
	 *
	 * @return $this
	 */
	public function getAlignmentItem( string $id = 'alignment_item', $key = false ): BlackdsnControl {


		$this->addChoose( $id, array(
			'label'   => __( 'Alignment Item', 'elementor' ),
			'options' => [
				'flex-start' => [
					'title' => __( 'Top', 'blackdsn' ),
					'icon'  => 'eicon-v-align-top',
				],
				'center'     => [
					'title' => __( 'Center', 'blackdsn' ),
					'icon'  => 'eicon-v-align-middle',
				],
				'flex-end'   => [
					'title' => __( 'Bottom', 'blackdsn' ),
					'icon'  => 'eicon-v-align-bottom',
				],
			],
			'default' => 'center',
		) );
		if ( $key ) {
			$this->getSelectorAlignItems( $key );
		}

		return $this;
	}

	/**
	 * @param $key
	 * @param string $value
	 *
	 * @return $this
	 */
	public function getSelectorAlignItems( $key, string $value = "" ): BlackdsnControl {
		return $this->setSelectors( $key, 'align-items:{{VALUE}};' . $value );

	}


	/**
	 * @param false $isResponsive
	 *
	 * @return $this
	 */
	public function get( bool $isResponsive = false ): BlackdsnControl {

		if ( $this->isGroup ) {
			$this->element_base->add_group_control( $this->id, $this->args );
		} elseif ( ! $isResponsive ) {
			$this->element_base->add_control( $this->id, $this->args );
		} else {
			$this->element_base->add_responsive_control( $this->id, $this->args );
		}

		$this->isGroup = false;
		$this->args    = [];


		return $this;
	}


	/**
	 * @return $this
	 */
	public function get_popover(): BlackdsnControl {
		$this->get();
		$this->element_base->start_popover();

		return $this;
	}


	/**
	 * @return $this
	 */
	public function end_popover(): BlackdsnControl {
		$this->element_base->end_popover();

		return $this;
	}


	/**
	 * @param $id
	 * @param array $args
	 *
	 * @return $this
	 */
	public function start_popover( $id, array $args = array() ): BlackdsnControl {

		$this->id     = $id;
		$args['type'] = Controls_Manager::POPOVER_TOGGLE;
		$this->args   = $args;

		return $this;
	}


	/**
	 * @return $this
	 */
	public function getResponsive(): BlackdsnControl {
		$this->element_base->add_responsive_control( $this->id, $this->args );
		$this->args = [];

		return $this;
	}

	public function getGroup(): BlackdsnControl {
		$this->element_base->add_group_control( $this->id, $this->args );
		$this->isGroup = false;
		$this->args    = [];

		return $this;
	}


	/**
	 * @param array $args
	 *
	 * @return array
	 */
	public function getOptionVerBackground( array $args = array() ): array {

		return array_merge( [
			''             => __( 'Default', 'blackdsn' ),
			'v-light'      => __( 'Light', 'blackdsn' ),
			'v-light-head' => __( 'Light (Static)', 'blackdsn' ),
			'v-dark'       => __( 'Dark', 'blackdsn' ),
			'v-dark-head'  => __( 'Dark  (Static)', 'blackdsn' ),
		], $args );

	}

	/**
	 * @param array $args
	 *
	 * @return array
	 */
	public function getOptionBackground( array $args = array() ): array {
		return array_merge( [
			'background-transparent' => __( 'Default', 'blackdsn' ),
			'background-main'        => __( 'Background Main', 'blackdsn' ),
			'background-section'     => __( 'Background Section', 'blackdsn' ),
			'background-theme'       => __( 'Background Theme', 'blackdsn' ),
		], $args );

	}


	/**
	 * @param $id
	 * @param array $args
	 *
	 * @return $this
	 */
	public function addIconColor( $id, array $args = array() ): BlackdsnControl {

		return $this->addSelect( $id, [
			''                         => __( 'custom', 'blackdsn' ),
			'dsn-icon-body-color'      => __( 'Body Color', 'blackdsn' ),
			'dsn-icon-heading-color'   => __( 'Heading Color', 'blackdsn' ),
			'dsn-icon-border-color'    => __( 'Border Color', 'blackdsn' ),
			'dsn-icon-main-color'      => __( 'Main Color', 'blackdsn' ),
			'dsn-icon-assistant-color' => __( 'Section Color', 'blackdsn' ),
			'dsn-icon-theme-color'     => __( 'Theme Color', 'blackdsn' ),
		], $args )
		            ->setDefault( 'dsn-icon-theme-color' )
		            ->setLabel( esc_html__( 'Color', 'elementor' ) );


	}

	/**
	 * @param string|array $post_type
	 *
	 * @return array
	 */
	public function getPostsArray( $post_type = 'any' ): array {
		$post_type = get_posts( array(
			'posts_per_page' => - 1,
			'post_type'      => $post_type,
		) );
		$items     = array();
		if ( count( $post_type ) ) {
			foreach ( $post_type as $po ):
				$items[ $po->ID ] = $po->post_title;
			endforeach;
		}

		return $items;
	}


	public function getArgs(): array {
		return $this->args;
	}


}