<?php

use Dsn\Element\Blackdsn_Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Widget_Base;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}


/**
 * Elementor accordion widget.
 *
 * Elementor widget that displays a collapsible display of content in an
 * accordion style, showing only one item at a time.
 *
 * @since 1.0.0
 */
class BlackdsnTitle extends Widget_Base {

	use Blackdsn_Widget_Base;


	/**
	 * Get widget name.
	 *
	 * Retrieve accordion widget name.
	 *
	 * @return string Widget name.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_name(): string {
		return 'dsn_title';
	}


	/**
	 * Get widget title.
	 *
	 * Retrieve accordion widget title.
	 *
	 * @return string Widget title.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_title(): string {
		return __( 'Blackdsn Title Section', 'blackdsn' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve accordion widget icon.
	 *
	 * @return string Widget icon.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_icon(): string {
		return 'eicon-post-title';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @return array Widget keywords.
	 * @since 2.1.0
	 * @access public
	 *
	 */
	public function get_keywords(): array {
		return array_merge( $this->dsn_keywords(), [ 'title', 'header' ] );
	}


	/**
	 * Register accordion widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {


		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Title', 'blackdsn' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);


		$control = $this->getControl();


		$control->addTextarea( 'title' )
		        ->setLabel( esc_html__( 'Title', 'blackdsn' ) )
		        ->setPlaceholder( esc_html__( 'Enter your title', 'blackdsn' ) )
		        ->setDefault( esc_html__( 'Add Your Title', 'blackdsn' ) )
		        ->setDynamicActive( true )
		        ->get();

		$control->addTextarea( 'description' )
		        ->setLabel( esc_html__( 'Description', 'blackdsn' ) )
		        ->setSeparator( "before" )
		        ->setPlaceholder( esc_html__( 'Enter your description', 'blackdsn' ) )
		        ->setDefault( esc_html__( 'Add Your Description', 'blackdsn' ) )
		        ->setDynamicActive( true )
		        ->get();

		$control->addChoose( 'direction' )
		        ->setLabel( esc_html__( 'Direction', 'blackdsn' ) )
		        ->setConditions( 'description', '!=', '' )
		        ->setOptionChoose( '1', __( 'After Title', 'elementor' ), 'eicon-v-align-top' )
		        ->setOptionChoose( '', __( 'Before Title', 'elementor' ), 'eicon-v-align-bottom' )
		        ->setDefault( '1' )
		        ->get();


		$control->addHtmlTag()
		        ->setSeparator( 'before' )
		        ->get();

		$control->addSize()
		        ->setDefault( 'title-h2' )
		        ->get();

		$this->end_controls_section();

		$this->styleTab();

	}


	private function styleTab() {
		$control = $this->getControl();

		$this->start_controls_section(
			'style_section',
			[
				'label' => esc_html__( 'Style Content', 'blackdsn' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$control->getAlign()
		        ->setDefault( "left" )
		        ->getResponsive();

		$control->getJustifyContent()
		        ->getSelectorjustifyContent( '.section-title' )
		        ->getResponsive();


		$control->addSelect( 'dsn_section_width_layout', [
			''                                  => __( 'Default', 'blackdsn' ),
			'dsn-container'                     => __( 'Wide Page', 'blackdsn' ),
			'container'                         => __( 'Container Page', 'blackdsn' ),
			'dsn-container dsn-right-container' => __( 'Right Container Page', 'blackdsn' ),
			'dsn-container dsn-left-container'  => __( 'Left Container Page', 'blackdsn' ),
		], [
			'label'        => __( 'Width Layout', 'blackdsn' ),
			'default'      => '',
			'prefix_class' => '',
		] )
		        ->setSeparator( "before" )
		        ->get();

		$control->addSlider( 'spaces_description' )
		        ->setLabel( __( 'Spacing Between', 'elementor' ) )
		        ->setRangePx( 0, 120 )
		        ->setSelectors( '.description.mb-10', 'margin-bottom: {{SIZE}}{{UNIT}};' )
		        ->setSelectors( '.description.mt-10', 'margin-top: {{SIZE}}{{UNIT}};' )
		        ->getResponsive();

		$control->addSwitcher( 'space_section' )
		        ->setLabel( esc_html__( "Use Space Default", 'blackdsn' ) )
		        ->setDefault( 'mb-70' )
		        ->setReturn_value( 'mb-70' )
		        ->get();

		$control->addHiddenNoSpace( 'hidden-space' )
		        ->setConditions( 'space_section', 'mb-70' )
		        ->get();

		$this->end_controls_section();


		$this->start_controls_section(
			'style_title_section',
			[
				'label' => esc_html__( 'Style Title', 'blackdsn' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$control->addHeadingColor( 'title_color', '.dsn-heading-title' )
		        ->setLabel( __( 'Custom Color', 'blackdsn' ) )
		        ->get();
		$control->addTypography( 'title_typography' )->getGroup();

		$control->addLineText( 'dsn_line_title' )
		        ->get();

		$control->addLineText( 'dsn_line_title_into' )
		        ->setLabel( __( 'Line Text Into', 'blackdsn' ) )
		        ->get();

		$control->addSlider( 'space', $control->getDefaultWidthHeight() )
		        ->setLabel( __( 'Max Width', 'elementor' ) )
		        ->setSelectors( '.dsn-heading-title', 'max-width: {{SIZE}}{{UNIT}};' )
		        ->getResponsive();


		$control->addTextShadow( 'title_Text_shadow' )->getGroup();
		$control->addBlendMode( 'title_blend_mode' )->get();

		$control->addSwitcher( 'use_as_stroke' )
		        ->setReturn_value( 'letter-stroke' )
		        ->setLabel( __( 'Use As Stroke', 'blackdsn' ) )
		        ->get();

		$control->addSelect( 'dsn_title_animate', [
			''         => __( 'None', 'blackdsn' ),
			'dsn-up'   => __( 'Fade', 'blackdsn' ),
			'dsn-text' => __( 'Text', 'blackdsn' ),
		] )
		        ->setLabel( __( 'Animate Text', 'blackdsn' ) )
		        ->setDefault( '' )
		        ->setSeparator( "before" )
		        ->get();


		$this->end_controls_section();


		$this->start_controls_section(
			'style_description_section',
			[
				'label' => esc_html__( 'Style Description', 'blackdsn' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$control->addHeadingColor( 'description_color', '.description' )
		        ->setLabel( __( 'Custom Color', 'blackdsn' ) )
		        ->get();
		$control->addTypography( 'description_typography', '.description' )->getGroup();

		$control->addLineText( 'dsn_line_description' )
		        ->setDefault( 'border-section-bottom' )
		        ->get();

		$control->addLineText( 'dsn_line_description_into' )
		        ->setDefault( '' )
		        ->setLabel( __( 'Line Text Into', 'blackdsn' ) )
		        ->get();

		$control->addSlider( 'space_description', $control->getDefaultWidthHeight() )
		        ->setLabel( __( 'Max Width', 'elementor' ) )
		        ->setSelectors( '.description', 'max-width: {{SIZE}}{{UNIT}};' )
		        ->getResponsive();


		$control->addSelect( 'dsn_description_animate', [
			''         => __( 'None', 'blackdsn' ),
			'dsn-up'   => __( 'Fade', 'blackdsn' ),
			'dsn-text' => __( 'Text', 'blackdsn' ),
		] )
		        ->setLabel( __( 'Animate Text', 'blackdsn' ) )
		        ->setDefault( '' )
		        ->setSeparator( "before" )
		        ->get();


		$this->end_controls_section();


	}

	/**
	 * Render accordion widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {

		$this->add_inline_editing_attributes( 'title' );
		$this->add_inline_editing_attributes( 'description' );

		blackdsn_render_widget_motion_effect( $this , 'section-title' );

		echo blackdsn_shortcode_render_group( 'title', array( 'widget-base' => $this ) );

	}


}
