<?php

use Elementor\Controls_Manager;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
};

/**
 * Elementor accordion widget.
 *
 * Elementor widget that displays a collapsible display of content in an
 * accordion style, showing only one item at a time.
 *
 * @since 1.0.0
 */
class  BlackdsnList extends \Dsn\Element\BlackdsnLayout {

	use \Dsn\Element\Blackdsn_Widget_Base;


	protected $range_col = 15;
	protected $range_row = 15;

	/**
	 * Get widget name.
	 *
	 * Retrieve accordion widget name.
	 *
	 * @return string Widget name.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_name() {
		return 'dsn_list';
	}


	/**
	 * Get widget title.
	 *
	 * Retrieve accordion widget title.
	 *
	 * @return string Widget title.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_title() {
		return __( 'Blackdsn List', 'blackdsn' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve accordion widget icon.
	 *
	 * @return string Widget icon.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_icon() {
		return 'eicon-bullet-list';

	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @return array Widget keywords.
	 * @since 2.1.0
	 * @access public
	 *
	 */
	public function get_keywords() {
		return array_merge( $this->dsn_keywords(), [ 'list' ] );
	}


	/**
	 * Register accordion widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {


		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Accordion', 'blackdsn' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);


		$control = $this->getControl();

		$this->getLayout();


		$control->startRepeater();


		$control->get_element_base()->add_control(
			'icon',
			[
				'label' => esc_html__( 'Icon', 'elementor' ),
				'type'  => \Elementor\Controls_Manager::ICONS,
				[
					'value'   => 'fas fa-check',
					'library' => 'fa-solid',
				],
			]
		);

		$control->addTextarea( 'text' )
		        ->setDefault( 'Web Design' )
		        ->setLabel( esc_html__( "Text", 'blackdsn' ) )
		        ->setLabelBlock()
		        ->setDynamicActive( true )
		        ->get();


		$control->endRepeater( 'items' )
		        ->setLabel( esc_html__( 'List Items', 'blackdsn' ) )
		        ->setTitle_field( 'elementor.helpers.renderIcon( this, icon, {}, "i", "panel" ) || \'<i class="{{ icon }}" aria-hidden="true"></i>\' }}} {{{ text' )
		        ->get();

		$control->addSize()
		        ->setSeparator( "before" )
		        ->get();
		$control->addHtmlTag()->setDefault( 'p' )->get();


		$this->end_controls_section();

		$this->styleTab();
	}


	private function styleTab() {
		$control = $this->getControl();

		$this->start_controls_section(
			'style_section',
			[
				'label' => esc_html__( 'Style', 'blackdsn' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$control->getJustifyContent( 'jus_con_list', '.list-item' )
		        ->getResponsive();

		$control->addSlider( 'icon_indent' )
		        ->setLabel( __( 'Icon Spacing', 'elementor' ) )
		        ->setRangePx( 0, 50 )
		        ->setSelectors( '.dsn-icon', 'margin-right: {{SIZE}}{{UNIT}};' )
		        ->getResponsive();

		$this->getGridSpace();
		$this->__style_controller( $control );

		$this->end_controls_section();

		$this->start_controls_section(
			'icon_section',
			[
				'label' => esc_html__( 'Icon Style', 'blackdsn' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);


		$control->addIconColor( 'icon_color_select' )
		        ->setPrefix_class()
		        ->get();


		$this->add_control(
			'icon_color',
			[
				'label'     => esc_html__( 'Color', 'elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'condition' => [
					'icon_color_select' => ''
				],
				'selectors' => [
					'{{WRAPPER}} .dsn-icon i'        => 'color: {{VALUE}};',
					'{{WRAPPER}} .dsn-icon svg'      => 'fill: {{VALUE}};',
					'{{WRAPPER}} .dsn-icon svg path' => 'fill: {{VALUE}};',
				],

			]
		);

		$control->addSlider( 'width_icon', $control->getDefaultWidthHeight() )
		        ->setLabel( __( 'Size Icon', 'elementor' ) )
		        ->setSelectors( '.dsn-list .dsn-icon:not(.icon-link)', '--dsn-icon-size: {{SIZE}}{{UNIT}};' )
		        ->getResponsive();


		$control->addPaddingGroup( 'item_padding_icon', '.dsn-icon i ,{{WRAPPER}} .dsn-icon  svg' )
		        ->setSeparator( "before" )
		        ->getResponsive();


		$control->addBorderRadiusGroup( 'item_border_radius_icon', '.dsn-icon i ,{{WRAPPER}} .dsn-icon svg' )
		        ->getResponsive();

		$control->addBorder( 'item_border_style_icon', '.dsn-icon i ,{{WRAPPER}} .dsn-icon  svg' )->getGroup();
		$control->addBoxShadow( 'item_box_shadow_icon', '.dsn-icon i ,{{WRAPPER}} .dsn-icon  svg' )->getGroup();


		$this->end_controls_section();

		$this->start_controls_section(
			'title_section',
			[
				'label' => esc_html__( 'Title Style', 'blackdsn' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);
		$control->addHeadingColor( 'title_color', '.dsn-heading-title' )
		        ->setLabel( __( 'Custom Color', 'blackdsn' ) )
		        ->get();

		$control->addSlider( 'space', $control->getDefaultWidthHeight() )
		        ->setLabel( __( 'Max Width', 'elementor' ) )
		        ->setSelectors( '.dsn-heading-title', 'max-width: {{SIZE}}{{UNIT}};' )
		        ->getResponsive();

		$control->addTypography( 'title_typography' )->getGroup();
		$control->addTextShadow( 'title_Text_shadow' )->getGroup();
		$this->end_controls_section();


	}

	public function __style_controller( BlackdsnControl $control ) {


		$control->addSelect( 'bg_ver_btn', $control->getOptionVerBackground() )
		        ->setLabel( esc_html__( 'Version Background', 'blackdsn' ) )
		        ->setSeparatorBefore()
		        ->setLabelBlock()
		        ->setDefault( '' )
		        ->get();

		$control->addSelect( 'bg_btn', $control->getOptionBackground() )
		        ->setLabel( esc_html__( 'Background', 'blackdsn' ) )
		        ->setLabelBlock()
		        ->setDefault( 'background-transparent' )
		        ->get();

		$control->addPaddingGroup( 'item_padding_service', '.list-item' )
		        ->setSeparator( "before" )
		        ->getResponsive();


		$control->addBorderRadiusGroup( 'item_border_radius', '.list-item' )
		        ->getResponsive();

		$control->addBorder( 'item_border_style', '.list-item' )->getGroup();
		$control->addBoxShadow( 'item_box_shadow', '.list-item' )->getGroup();
	}


	/**
	 * Render accordion widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$control = new blackdsnShortCode( array( 'widget-base' => $this ) );

		$items_key = $this->getKeys( 'items', [ 'text' ] );

		$this->addPrefixClassLayout( 'dsn-list', $control );
		$this->addAnimateFade( 'list-item', $control );
		$this->add_render_attribute( 'dsn-list', 'class', 'dsn-list' );

		echo '<ul ' . $this->get_render_attribute_string( 'dsn-list' ) . '>';
		echo blackdsn_shortcode_render_group( 'list', array( 'widget-base' => $this ), $items_key );
		echo '</ul>';

	}


}
