<?php

use Dsn\Element\Blackdsn_Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Css_Filter;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Image_Size;
use Elementor\Widget_Base;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}


/**
 * Elementor accordion widget.
 *
 * Elementor widget that displays a collapsible display of content in an
 * accordion style, showing only one item at a time.
 *
 * @since 1.0.0
 */
class BlackdsnImage extends Widget_Base {

	use Blackdsn_Widget_Base;


	/**
	 * Get widget name.
	 *
	 * Retrieve accordion widget name.
	 *
	 * @return string Widget name.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_name(): string {
		return 'dsn_image';
	}


	/**
	 * Get widget title.
	 *
	 * Retrieve accordion widget title.
	 *
	 * @return string Widget title.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_title(): string {
		return __( 'Parallax Image', 'blackdsn' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve accordion widget icon.
	 *
	 * @return string Widget icon.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_icon(): string {
		return 'eicon-image-rollover';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @return array Widget keywords.
	 * @since 2.1.0
	 * @access public
	 *
	 */
	public function get_keywords(): array {
		return array_merge( $this->dsn_keywords(), [ 'parallax', 'image' ] );
	}


	/**
	 * Register accordion widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {


		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Content', 'blackdsn' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);


		$control = $this->getControl();
		$control->getGroupImage();


		$this->getOverlay();

		$control->addBlendMode( 'bland_overlay', '.img-box-parallax img' )
		        ->get();


		$control->addPopover( 'position_image' )
		        ->setLabel( __( 'Position Image', 'blackdsn' ) )
		        ->get();
		$this->start_popover();

		$control->addNumberSlider( 'position_image_position_x', 0, 100, 1 )
		        ->setLabel( esc_html__( 'Position X', 'blackdsn' ) )
		        ->setDefaultRange( 50 )
		        ->setSelectors( 'img', 'object-position:{{SIZE}}% {{position_image_position_y.SIZE}}%' )
		        ->getResponsive();

		$control->addNumberSlider( 'position_image_position_y', 0, 100, 1 )
		        ->setLabel( esc_html__( 'Position Y', 'blackdsn' ) )
		        ->setDefaultRange( 50 )
		        ->setSelectors( 'img', 'object-position:{{position_image_position_x.SIZE}}% {{SIZE}}%' )
		        ->getResponsive();


		$this->end_popover();

		$this->add_control(
			'view',
			[
				'label'   => __( 'View', 'elementor' ),
				'type'    => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);


		$this->end_controls_section();
		$this->getEntranceAnimationImage();


		$this->popupImage();

		$this->styleTab();

	}


	protected function popupImage() {

		$this->start_controls_section(
			'dsn_option_popup',
			[
				'label' => __( 'Popup Image', 'blackdsn' ),
			]
		);
		$this->add_control(
			'use_as_popup',
			[
				'label'        => __( 'Show On Popup', 'blackdsn' ),
				'type'         => Controls_Manager::SWITCHER,
				'return_value' => '1',
				'default'      => '',
			]
		);
		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name'      => 'size_img_popup',
				'default'   => 'large',
				'separator' => 'none',
				'condition' => [ 'use_as_popup' => '1' ],

			]
		);

		$this->getControl()
		     ->addText( 'group_by_fancybox' )
		     ->setLabel( esc_html__( 'Group By', 'blackdsn' ) )
		     ->setConditions( 'use_as_popup', '1' )
		     ->get();


		$this->end_controls_section();
	}

	private function getOverlay() {
		$control = $this->getControl();

		$control->addPopover( 'opacity_overlay_popover' )
		        ->setLabel( __( "Background Overlay", 'elementor' ) )
		        ->setSeparator( "before" )
		        ->get();

		$this->start_popover();

		$control->addColor( 'color_overlay' )
		        ->setLabel( __( 'Color Overlay' ) )
		        ->get();

		$control->addNumberSlider( 'opacity_overlay', 0, 10, 1 )
		        ->setLabel( __( 'Opacity Overlay' ) )
		        ->setDefaultRange( 0 )
		        ->get();

		$this->end_popover();

	}

	private function getEntranceAnimationImage() {

		$this->start_controls_section(
			'dsn_option_motion_effects',
			[
				'label' => __( 'Image Motion Effects', 'blackdsn' ),
			]
		);
		$control = $this->getControl();

		$control->addSwitcher( "use_scrolling_effects" )
		        ->setSeparator( "before" )
		        ->setLabel( esc_html__( 'Scrolling Effects', 'blackdsn' ) )
		        ->setDefault( "1" )
		        ->get();


		$control->addChoose( 'effect_open_image' )
		        ->setLabel( __( "Effect Open Image", 'blackdsn' ) )
		        ->setOptionChoose( '', __( 'None', 'blackdsn' ), 'eicon-minus-square-o' )
		        ->setOptionChoose( 'dsn-animate dsn-effect-up', __( 'Up', 'blackdsn' ), 'eicon-arrow-up' )
		        ->setOptionChoose( 'dsn-animate dsn-effect-right', __( 'Right', 'blackdsn' ), 'eicon-arrow-right' )
		        ->setOptionChoose( 'dsn-animate dsn-effect-down', __( 'Down', 'blackdsn' ), 'eicon-arrow-down' )
		        ->setOptionChoose( 'dsn-animate dsn-effect-left', __( 'Left', 'blackdsn' ), 'eicon-arrow-left' )
		        ->setConditions( 'use_scrolling_effects', '1' )
		        ->setDefault( '' )
		        ->setToggle()
		        ->get();


		$control->addHidden( 'hidden_effect_image' )
		        ->setDefault( 'over-hidden' )
		        ->setPrefix_class()
		        ->setConditions( 'effect_open_image', '!=', '', 'and' )
		        ->setConditions( 'use_scrolling_effects', '==', '1', 'and' )
		        ->get();


		$control->addChoose( 'direction_animate_image' )
		        ->setLabel( __( "Direction", 'blackdsn' ) )
		        ->setOptionChoose( 'has-opposite-direction', __( 'Up', 'blackdsn' ), 'eicon-arrow-up' )
		        ->setOptionChoose( 'has-direction', __( 'Down', 'blackdsn' ), 'eicon-arrow-down' )
		        ->setDefault( 'has-direction' )
		        ->setConditions( 'use_scrolling_effects', '1' )
		        ->get();

		$control->addSelect( 'animate_image_style', [
			''                 => __( "Default", 'blackdsn' ),
			'has-bigger-scale' => __( "Scale Down", 'blackdsn' ),
			'has-scale'        => __( "Scale Up", 'blackdsn' ),
		] )
		        ->setLabel( __( "Entrance Animation Type", 'blackdsn' ) )
		        ->setConditions( 'use_scrolling_effects', '1' )
		        ->setDefault( '' )->get();


		$control->getTriggerHock( 'animate_image_triggerhook' )
		        ->setConditions( 'use_scrolling_effects', '1' )
		        ->get();


		$control->addNumberSlider( 'speed_animation_image', 0, 100, 10 )
		        ->setDefaultRange( 30 )
		        ->setLabel( __( 'Speed', 'blackdsn' ) )
		        ->setConditions( 'use_scrolling_effects', '1' )
		        ->get();

		$this->mouseEffects();
		$this->end_controls_section();

	}

	private function mouseEffects() {
		$control = $this->getControl();

		$control->addSwitcher( "mouse_effects" )
		        ->setLabel( esc_html__( 'Mouse Effects', 'blackdsn' ) )
		        ->setSeparator( "before" )
		        ->get();

		$control->addSwitcher( "zoom" )
		        ->setLabel( esc_html__( 'Zoom Button', 'blackdsn' ) )
		        ->setReturn_value( 'image-zoom' )
		        ->setConditions( 'mouse_effects', '1' )
		        ->get();


	}

	private function styleTab() {
		$control = $this->getControl();

		$this->start_controls_section(
			'style_image_section',
			[
				'label' => __( 'Image', 'elementor' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$control->addSwitcher( 'dsn_equal_height', [
			'label'        => __( 'Enable Equal Height', 'blackdsn' ),
			'prefix_class' => 'dsn-equal-height ',
			'default'      => '',
		] )
		        ->setReturn_value( 'h-100' )
		        ->get();

		$control->addSelect( 'custom_position', [
			'inherit'  => __( 'inherit', 'blackdsn' ),
			'relative' => __( 'relative', 'blackdsn' ),
			'absolute' => __( 'absolute', 'blackdsn' ),
		], [
			'label'         => __( 'Position', 'elementor' ),
			'default'       => 'relative',
			'selectors'     => [
				'{{WRAPPER}} ' => 'position: {{VALUE}};',
			],
			'hide_in_inner' => true,
		] )
		        ->setPrefix_class( 'dsn-position%s-' )
		        ->getResponsive();


		$control->addSlider( 'width', $control->getDefaultWidthHeight() )
		        ->setLabel( __( 'Width', 'elementor' ) )
		        ->setSelectors( '.img-box-parallax', 'width: {{SIZE}}{{UNIT}};' )
		        ->getResponsive();

		$control->addSlider( 'space', $control->getDefaultWidthHeight() )
		        ->setLabel( __( 'Max Width', 'elementor' ) )
		        ->setSelectors( '.img-box-parallax', 'max-width: {{SIZE}}{{UNIT}};' )
		        ->getResponsive();

		$control->addSlider( 'height', $control->getDefaultWidthHeight( 'vh' ) )
		        ->setLabel( __( 'Height', 'elementor' ) )
		        ->setSelectors( '.img-box-parallax', 'height: {{SIZE}}{{UNIT}};' )
		        ->getResponsive();

		$control->getAlign()->getResponsive();


		$control->addSelect( 'object-fit', [
			''        => __( 'Default', 'elementor' ),
			'fill'    => __( 'Fill', 'elementor' ),
			'cover'   => __( 'Cover', 'elementor' ),
			'contain' => __( 'Contain', 'elementor' ),
		], [
			'label' => __( 'Object Fit', 'elementor' ),

			'options'   => [
				''        => __( 'Default', 'elementor' ),
				'fill'    => __( 'Fill', 'elementor' ),
				'cover'   => __( 'Cover', 'elementor' ),
				'contain' => __( 'Contain', 'elementor' ),
			],
			'default'   => '',
			'selectors' => [
				'{{WRAPPER}} .img-box-parallax img' => 'object-fit: {{VALUE}};',
			],
		] )->setSeparator( "before" )->get();

		$this->hoverImage();


		$this->end_controls_section();


	}

	private function hoverImage() {


		$this->start_controls_tabs( 'image_effects' );

		$this->start_controls_tab( 'normal',
			[
				'label' => __( 'Normal', 'elementor' ),
			]
		);

		$this->add_control(
			'opacity',
			[
				'label'     => __( 'Opacity', 'elementor' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'max'  => 1,
						'min'  => 0.10,
						'step' => 0.01,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .img-box-parallax ' => 'opacity: {{SIZE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Css_Filter::get_type(),
			[
				'name'     => 'css_filters',
				'selector' => '{{WRAPPER}} .img-box-parallax ',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab( 'hover',
			[
				'label' => __( 'Hover', 'elementor' ),
			]
		);

		$this->add_control(
			'opacity_hover',
			[
				'label'     => __( 'Opacity', 'elementor' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'max'  => 1,
						'min'  => 0.10,
						'step' => 0.01,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .img-box-parallax:hover ' => 'opacity: {{SIZE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Css_Filter::get_type(),
			[
				'name'     => 'css_filters_hover',
				'selector' => '{{WRAPPER}} .img-box-parallax:hover ',
			]
		);

		$this->add_control(
			'background_hover_transition',
			[
				'label'     => __( 'Transition Duration', 'elementor' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'max'  => 3,
						'step' => 0.1,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .img-box-parallax ' => 'transition: {{SIZE}}s;',
				],
			]
		);

		$this->add_control(
			'hover_animation',
			[
				'label' => __( 'Hover Animation', 'elementor' ),
				'type'  => Controls_Manager::HOVER_ANIMATION,
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();


		$this->add_responsive_control(
			'image_border_radius',
			[
				'label'      => __( 'Border Radius', 'elementor' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'separator'  => 'before',
				'selectors'  => [
					'{{WRAPPER}} .img-box-parallax  ' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'image_box_shadow',
				'exclude'  => [
					'box_shadow_position',
				],
				'selector' => '{{WRAPPER}} .img-box-parallax , {{WRAPPER}} .img-box-parallax img',
			]
		);
	}

	/**
	 * Render accordion widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$this->add_inline_editing_attributes( 'caption' );
		blackdsn_render_widget_motion_effect( $this, 'parallax' );

		echo blackdsn_shortcode_render_group( 'image', array( 'widget-base' => $this ) );
	}


	public function get_style_depends() {
		return $this->getDepends();
	}

	public function get_script_depends() {
		return $this->getDepends();
	}

	private function getDepends() {

		if ( blackdsn_is_built_with_elementor() ) {
			return [ blackdsnFrontEnd::FANCYBOX ];
		}
		$out       = [];
		$shortcode = $this->getShortCode();
		if ( $shortcode->getVal( 'use_as_popup' ) ) {
			$out[] = blackdsnFrontEnd::FANCYBOX;
		}

		return $out;
	}

}
