<?php

use Dsn\Element\Blackdsn_Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Widget_Base;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}


/**
 * Elementor accordion widget.
 *
 * Elementor widget that displays a collapsible display of content in an
 * accordion style, showing only one item at a time.
 *
 * @since 1.0.0
 */
class BlackdsnHeading extends Widget_Base {

	use Blackdsn_Widget_Base;


	/**
	 * Get widget name.
	 *
	 * Retrieve accordion widget name.
	 *
	 * @return string Widget name.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_name(): string {
		return 'dsn_heading';
	}

	public function get_custom_help_url(): string {
		return 'https://developers.elementor.com/docs/widgets/';
	}


	/**
	 * Get widget title.
	 *
	 * Retrieve accordion widget title.
	 *
	 * @return string Widget title.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_title(): string {
		return __( 'blackdsn Heading', 'blackdsn' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve accordion widget icon.
	 *
	 * @return string Widget icon.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_icon(): string {
		return 'eicon-t-letter';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @return array Widget keywords.
	 * @since 2.1.0
	 * @access public
	 *
	 */
	public function get_keywords(): array {
		return array_merge( $this->dsn_keywords(), [ 'information', 'letter', 'heading', 'text', 'paragraph' ] );
	}


	/**
	 * Register accordion widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */

	protected function register_controls() {


		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Content', 'blackdsn' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);


		$control = $this->getControl();

		$control->addTextarea( 'title' )
		        ->setLabel( esc_html__( 'Title', 'blackdsn' ) )
		        ->setPlaceholder( esc_html__( 'Enter your title', 'blackdsn' ) )
		        ->setDefault( esc_html__( 'Add Your Heading Text Here', 'blackdsn' ) )
		        ->get();

		$control->addLink( 'link' )
		        ->setLabel( esc_html__( 'Link', 'blackdsn' ) )
		        ->setSeparator( "before" )
		        ->get();

		$control->addSize()
		        ->setSeparator( "before" )
		        ->get();


		$control->addHtmlTag()->get();
		$control->addLineText()
		        ->get();

		$control->getAlign()->getResponsive();

		$control->addSelect( 'dsn_title_animate', [
			''         => __( 'None', 'blackdsn' ),
			'dsn-up'   => __( 'Fade', 'blackdsn' ),
			'dsn-text' => __( 'Text', 'blackdsn' ),
		] )
		        ->setLabel( __( 'Animate Text', 'blackdsn' ) )
		        ->setDefault( '' )->setSeparator( "before" )
		        ->get();

		$control->addSwitcher( 'animate_number' )
		        ->setLabel( __( 'Animate Number', 'blackdsn' ) )
		        ->setReturn_value( "dsn-animate-number" )
		        ->get();

		$this->end_controls_section();

		$this->styleTab();
		$this->TitleCover();

	}


	private function styleTab() {
		$control = $this->getControl();

		$this->start_controls_section(
			'style_section',
			[
				'label' => esc_html__( 'Style', 'blackdsn' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$control->addSelect( 'bg_ver_btn', $control->getOptionVerBackground() )
		        ->setLabel( esc_html__( 'Version Background', 'blackdsn' ) )
		        ->setDefault( '' )
		        ->setPrefix_class()
		        ->get();

		$control->addSelect( 'bg_btn', $control->getOptionBackground() )
		        ->setLabel( esc_html__( 'Background Button', 'blackdsn' ) )
		        ->setDefault( 'background-transparent' )
		        ->setPrefix_class()
		        ->get();

		$control->addImage()
		        ->setDefault( [] )
		        ->setSeparatorBefore()
		        ->setSelectors( '.dsn-heading-title', 'background-image: url("{{URL}}");color: transparent;-webkit-text-stroke: 1.5px var(--heading-color);-webkit-text-fill-color: transparent;-webkit-background-clip: text;background-clip: text;transform: scale3d(1, 1.2, 1);background-position: center;
background-repeat: no-repeat;
background-size: cover;' )
		        ->getResponsive();

		$control->addPaddingGroup( 'item_clip_path', '.dsn-heading-title', [ 'selectors' => [] ] )
		        ->setSelectors( '.dsn-heading-title',
			        'clip-path:inset({{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}})' )
		        ->setSeparator( "both" )
		        ->setLabel( esc_html__( "Clip Path", 'blackdsn' ) )
		        ->getResponsive();

		$control->addHeadingColor( 'title_color', '.dsn-heading-title' )
		        ->setLabel( __( 'Custom Color', 'blackdsn' ) )
		        ->get();

		$control->addSlider( 'space', $control->getDefaultWidthHeight() )
		        ->setLabel( __( 'Max Width', 'elementor' ) )
		        ->setSelectors( '.dsn-heading-title', 'max-width: {{SIZE}}{{UNIT}};' )
		        ->getResponsive();

		$control->addTypography( 'title_typography' )->getGroup();
		$control->addTextShadow( 'title_Text_shadow' )->getGroup();
		$control->addBlendMode( 'title_blend_mode' )->get();

		$control->addSwitcher( 'use_as_stroke' )
		        ->setReturn_value( 'letter-stroke' )
		        ->setLabel( __( 'Use As Stroke', 'blackdsn' ) )
		        ->get();


		$this->end_controls_section();


	}

	private function TitleCover() {
		$control = $this->getControl();

		$this->start_controls_section(
			'section_title_cover',
			[
				'label' => esc_html__( 'Title Cover Option', 'blackdsn' ),
				'tab'   => Controls_Manager::TAB_SETTINGS,
			]
		);

		$control->addSwitcher( 'use_as_title_cover' )
		        ->setLabel( __( 'Use As Title cover', 'blackdsn' ) )
		        ->setReturn_value( '1' )
		        ->get();

		$control->addHiddenNoSpace( 'hidden-space' )
		        ->setConditions( 'use_as_title_cover', '1' )
		        ->get();


		$control->addNumber( 'optcity_from', 0, 1, 0.1 )
		        ->setLabel( __( 'Opacity From', 'blackdsn' ) )
		        ->setDefault( 0 )
		        ->setSelectors( '.dsn-move-section', 'opacity : {{VALUE}}' )
		        ->setConditions( 'use_as_title_cover', '1' )
		        ->get();


		$control->addNumber( 'opacity_to', 0, 1, 0.1 )
		        ->setLabel( __( 'Opacity To', 'blackdsn' ) )
		        ->setDefault( 0.07 )
		        ->setConditions( 'use_as_title_cover', '1' )
		        ->get();


		$control->addText( 'move_from_duration' )
		        ->setLabel( __( 'Move From', 'blackdsn' ), true )
		        ->setDefault( '-100%' )
		        ->setDescription( __( 'Move Section From.', 'blackdsn' ) )
		        ->setConditions( 'use_as_title_cover', '1' )
		        ->setSeparatorBefore()
		        ->get();

		$control->addText( 'move_to_duration' )
		        ->setLabel( __( 'Move To', 'blackdsn' ), true )
		        ->setDefault( '0%' )
		        ->setDescription( __( 'Move Section To.', 'blackdsn' ) )
		        ->setConditions( 'use_as_title_cover', '1' )
		        ->get();


		$control->getTriggerHock( 'triggerhook_duration' )
		        ->setConditions( 'use_as_title_cover', '1' )
		        ->setSeparatorBefore()
		        ->get();

		$control->addSwitcher( 'stop_on_mobile' )
		        ->setLabel( __( 'Stop On Mobile', 'blackdsn' ) )
		        ->setReturn_value( '1' )
		        ->setConditions( 'use_as_title_cover', '1' )
		        ->get();


		$this->end_controls_section();
	}

	/**
	 * Render accordion widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {

		$this->add_inline_editing_attributes( 'title' );
		echo blackdsn_shortcode_render_group( 'heading', array( 'widget-base' => $this ) );

	}

	protected function content_template() {
		echo blackdsn_shortcode_js_group( 'heading', array(), '' );
	}


}
