<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
};

/**
 * Elementor accordion widget.
 *
 * Elementor widget that displays a collapsible display of content in an
 * accordion style, showing only one item at a time.
 *
 * @since 1.0.0
 */
class BlackdsnGridMasonry extends \Dsn\Element\BlackdsnLayout {

	use \Dsn\Element\Blackdsn_Widget_Base;


	protected $range_col = 15;
	protected $range_row = 15;

	/**
	 * Get widget name.
	 *
	 * Retrieve accordion widget name.
	 *
	 * @return string Widget name.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_name() {
		return 'dsn-grid-masonry';
	}


	/**
	 * Get widget title.
	 *
	 * Retrieve accordion widget title.
	 *
	 * @return string Widget title.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_title() {
		return __( 'Blackdsn Grid Masonry', 'blackdsn' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve accordion widget icon.
	 *
	 * @return string Widget icon.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_icon() {
		return 'eicon-bullet-list';

	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @return array Widget keywords.
	 * @since 2.1.0
	 * @access public
	 *
	 */
	public function get_keywords() {
		return array_merge( $this->dsn_keywords(), [ 'grid', 'masonry' ] );
	}


	/**
	 * Register accordion widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _register_controls() {


		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'GridMasonry', 'blackdsn' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);


		$control = $this->getControl();

		$this->getLayout();

		$control->addSwitcher( 'list_with_number' )
		        ->setLabel( esc_html__( "Title With Number", 'blackdsn' ) )
		        ->setReturn_value( 'box-grid-number' )
		        ->setPrefix_class()
		        ->get();


		$control->startRepeater();

		$control->getGroupImage();


		$control->addTextarea( 'title' )
		        ->setLabel( esc_html__( 'Title', 'blackdsn' ) )
		        ->setPlaceholder( esc_html__( 'Enter your title', 'blackdsn' ) )
		        ->setDefault( esc_html__( 'Add Your Heading Text Here', 'blackdsn' ) )
		        ->get();

		$control->addLink( 'link' )
		        ->setLabel( esc_html__( 'Link', 'blackdsn' ) )
		        ->setSeparator( "before" )
		        ->get();

		$control->endRepeater( 'items' )
		        ->setLabel( esc_html__( 'List Items', 'blackdsn' ) )
		        ->setTitle_field( 'title' )
		        ->get();

		$control->addSize()
		        ->setSeparator( "before" )
		        ->setDefault( 'sm-title-block' )
		        ->get();

		$control->addLineText()->get();
		$control->addLineText( 'dsn_line_text_2' )
		        ->setLabel( esc_html__( 'Line Text Into', 'blackdsn' ) )
		        ->get();

		$control->addHtmlTag()->get();
		$control->getAlign()->getResponsive();


		$this->end_controls_section();

		$this->styleTab();
	}


	private function styleTab() {
		$control = $this->getControl();

		$this->start_controls_section(
			'style_section',
			[
				'label' => esc_html__( 'Style', 'blackdsn' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);
		$this->getGridSpace();


		$control->addSelect( 'bg_ver_btn', $control->getOptionVerBackground() )
		        ->setLabel( esc_html__( 'Version Background Service', 'blackdsn' ) )
		        ->setLabelBlock()
		        ->setDefault( '' )
		        ->setSeparatorBefore()
		        ->get();

		$control->addSelect( 'bg_btn', $control->getOptionBackground() )
		        ->setLabel( esc_html__( 'Background Service', 'blackdsn' ) )
		        ->setLabelBlock()
		        ->setDefault( 'background-section' )
		        ->get();


		$control->addPaddingGroup( 'item_padding', '.box-item' )
		        ->setSeparator( "before" )
		        ->getResponsive();


		$control->addBorderRadiusGroup( 'item_border_radius', '.box-item' )
		        ->getResponsive();

		$control->addBorder( 'item_border_style', '.box-item' )->getGroup();
		$control->addBoxShadow( 'item_box_shadow', '.box-item' )->getGroup();

		$this->end_controls_section();
		$this->styleImage();
		$this->getStyleContent();
	}


	private function styleImage() {
		$control = $this->getControl();

		$this->start_controls_section(
			'style_section-image',
			[
				'label' => esc_html__( 'Style Image', 'blackdsn' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$control->addSlider( 'height', $control->getDefaultWidthHeight( 'vh' ) )
		        ->setLabel( __( 'Height', 'elementor' ) )
		        ->setSelectors( '.box-item img', 'height: {{SIZE}}{{UNIT}};' )
		        ->getResponsive();


		$control->addPaddingGroup( 'item_padding_img', '.box-item img' )
		        ->setSeparator( "before" )
		        ->getResponsive();


		$control->addBorderRadiusGroup( 'item_border_radius_img', '.box-item img' )
		        ->getResponsive();

		$control->addBorder( 'item_border_style_img', '.box-item img' )->getGroup();
		$control->addBoxShadow( 'item_box_shadow_img', '.box-item img' )->getGroup();

		$this->end_controls_section();


	}


	private function getStyleContent() {
		$control = $this->getControl();


		$args = array(
			'dsn-title' => esc_html__( 'Title', 'blackdsn' )
		);

		foreach ( $args as $id => $value ):

			$this->start_controls_section(
				'style_content_service_' . $id,
				[
					'label' => $value,
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);


			$control->addColor( 'color_content_' . $id )
			        ->setLabel( esc_html__( "Color", 'blackdsn' ) )
			        ->setSeparator( "before" )
			        ->setSelectors( '.' . $id, 'color:{{VALUE}};' )
			        ->get();

			$control->addMarginGroup( 'item_margin_title', '.' . $id )
			        ->setSeparator( "before" )
			        ->getResponsive();

			$control->addTypography( 'item_typo_content_' . $id, '.' . $id )
			        ->getGroup();


			$control->addTextShadow( 'text_content_shadow_' . $id, '.' . $id )
			        ->getGroup();

			$this->end_controls_section();


		endforeach;


	}

	/**
	 * Render accordion widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$control = new blackdsnShortCode( array( 'widget-base' => $this ) );

		$items_key = $this->getKeys( 'items', [ 'title' ] );

		$this->addPrefixClassLayout( 'dsn-grid-masonry', $control );
		$this->addAnimateFade( 'box-item', $control );
		$this->add_render_attribute( 'dsn-grid-masonry', 'class', 'dsn-grid-masonry' );

		echo '<div ' . $this->get_render_attribute_string( 'dsn-grid-masonry' ) . '>';
		echo blackdsn_shortcode_render_group( 'grid-masonry', array( 'widget-base' => $this ), $items_key );
		echo '</div>';

	}


}
