<?php

use Dsn\Element\Blackdsn_Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Widget_Base;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}


/**
 * Elementor accordion widget.
 *
 * Elementor widget that displays a collapsible display of content in an
 * accordion style, showing only one item at a time.
 *
 * @since 1.0.0
 */
class BlackdsnButton extends Widget_Base {

	use Blackdsn_Widget_Base;


	/**
	 * Get widget name.
	 *
	 * Retrieve accordion widget name.
	 *
	 * @return string Widget name.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_name() {
		return 'dsn_button';
	}


	/**
	 * Get widget title.
	 *
	 * Retrieve accordion widget title.
	 *
	 * @return string Widget title.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_title() {
		return __( 'Blackdsn Button', 'blackdsn' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve accordion widget icon.
	 *
	 * @return string Widget icon.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_icon() {
		return 'eicon-button';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @return array Widget keywords.
	 * @since 2.1.0
	 * @access public
	 *
	 */
	public function get_keywords() {
		return array_merge( $this->dsn_keywords(), [ 'Button' ] );
	}


	/**
	 * Register accordion widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {


		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Content', 'blackdsn' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);


		$control = $this->getControl();
		$control->addSelect( 'style_btn', $control->getStyleButton() )
		        ->setLabel( esc_html__( 'Style', 'blackdsn' ) )
		        ->setDefault( 'dsn-def-btn' )
		        ->setPrefix_class()
		        ->get();


		$control->addSelect( 'bg_ver_btn', $control->getOptionVerBackground() )
		        ->setLabel( esc_html__( 'Version Background', 'blackdsn' ) )
		        ->setDefault( '' )
		        ->get();

		$control->addSelect( 'bg_btn', $control->getOptionBackground() )
		        ->setLabel( esc_html__( 'Background Button', 'blackdsn' ) )
		        ->setDefault( 'background-section' )
		        ->get();

		$control->addText( "text" )
		        ->setLabel( __( 'Text', 'elementor' ) )
		        ->setDynamicActive( true )
		        ->setDefault( __( 'Click here', 'elementor' ) )
		        ->setPlaceholder( __( 'Click here', 'elementor' ) )
		        ->setSeparator( "before" )
		        ->get();


		$control->addSwitcher( 'type_link' )
		        ->setLabel( esc_html__( "Type Link", 'blackdsn' ) )
		        ->setLabel_on( esc_html__( 'in', 'blackdsn' ) )
		        ->setLabel_off( esc_html__( 'out', 'blackdsn' ) )
		        ->get();


		$control->addLink( 'link' )
		        ->setLabel( esc_html__( 'Link', 'blackdsn' ) )
		        ->setDefault_url( '#0' )
		        ->setDefault_is_external( true )
		        ->setDefault_is_nofollow( true )
		        ->setDynamicActive( true )
		        ->setCondition( [ 'type_link!' => '1' ] )
		        ->get();


		$control->addSelect2( 'link_post_id', $control->getPostsArray() )
		        ->setLabel( esc_html__( 'Choose Page', 'blackdsn' ) )
		        ->setLabelBlock()
		        ->setCondition( [ 'type_link' => '1' ] )
		        ->get();

		$control->addSwitcher( "with_ajax" )
		        ->setLabel( esc_html__( 'With Ajax', 'blackdsn' ) )
		        ->setReturn_value( "effect-ajax" )
		        ->setCondition( [ 'type_link' => '1' ] )
		        ->get();

		$control->addText( "text_with_ajax" )
		        ->setLabel( esc_html__( 'Text With Ajax', 'blackdsn' ) )
		        ->setCondition( [ 'type_link' => '1', 'with_ajax' => 'effect-ajax' ] )
		        ->get();


		$this->add_control(
			'icon',
			[
				'label' => __( 'Icon', 'elementor' ),
				'type'  => Controls_Manager::ICONS,
			]
		);

		$control->addSelect( 'icon_align', [
			'left'  => __( 'Before', 'elementor' ),
			'right' => __( 'After', 'elementor' ),
		] )->setDefault( 'left' )
		        ->setLabel( __( 'Icon Position', 'elementor' ) )
		        ->setCondition( [ 'icon[value]!' => '', ] )
		        ->get()
		        ->addSlider( 'icon_indent' )
		        ->setLabel( __( 'Icon Spacing', 'elementor' ) )
		        ->setRangePx( 0, 50 )
		        ->setSelectors( '.dsn-btn .btn-icon-right', 'margin-left: {{SIZE}}{{UNIT}};' )
		        ->setSelectors( '.dsn-btn .btn-icon-left', 'margin-right: {{SIZE}}{{UNIT}};' )
		        ->get();

		$control->addHidden( 'view', [
			'label'   => __( 'View', 'elementor' ),
			'default' => 'traditional',
		] )->get();


		$control->addSwitcher( "full_width" )
		        ->setLabel( esc_html__( 'Full Width', 'blackdsn' ) )
		        ->setReturn_value( "w-100" )
		        ->setSeparator( "before" )
		        ->get();
		$control->getJustifyContent()
		        ->getSelectorjustifyContent( ' ', 'display:flex;' )
		        ->getResponsive();


		$control->addSwitcher( "popup_vid" )
		        ->setLabel( esc_html__( 'Use As Popup Video', 'blackdsn' ) )
		        ->setReturn_value( "vid" )
		        ->setSeparator( "before" )
		        ->get();


		$control->addSwitcher( "parallax" )
		        ->setLabel( esc_html__( 'Mouse Button Parallax', 'blackdsn' ) )
		        ->setSeparator( "before" )
		        ->get();

		$control->addSwitcher( "zoom" )
		        ->setLabel( esc_html__( 'Zoom Button', 'blackdsn' ) )
		        ->setReturn_value( 'image-zoom' )
		        ->setConditions( 'parallax', '1' )
		        ->get();

		$control->addSwitcher( "move_circle" )
		        ->setLabel( esc_html__( 'Move Bg', 'blackdsn' ) )
		        ->setReturn_value( 'move-circle' )
		        ->setConditions( 'parallax', '1' )
		        ->get();


		$control->addText( 'button_css_id',
			[ 'title' => __( 'Add your custom id WITHOUT the Pound key. e.g: my-id', 'elementor' ), ] )
		        ->setLabel( __( 'Button ID', 'elementor' ) )
		        ->setDynamicActive( true )
		        ->setDescription( __( 'Please make sure the ID is unique and not used elsewhere on the page this form is displayed. This field allows <code>A-z 0-9</code> & underscore chars without spaces.',
			        'elementor' ) )
		        ->setSeparator( "before" )
		        ->get();


		$this->end_controls_section();

		$this->styleTab();

	}


	private function styleTab() {

		$control = $this->getControl();

		$btn       = '.dsn-btn';
		$link_text = '.title-btn';


		$this->start_controls_section(
			'section_style',
			[
				'label' => __( 'Button', 'elementor' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);


		$control->addSlider( 'width', $control->getDefaultWidthHeight() )
		        ->setLabel( __( 'Width', 'elementor' ) )
		        ->setSelectors( $btn, 'width: {{SIZE}}{{UNIT}};' )
		        ->getResponsive();


		$control->addSlider( 'height', $control->getDefaultWidthHeight() )
		        ->setLabel( __( 'Height', 'elementor' ) )
		        ->setSelectors( $btn, 'height: {{SIZE}}{{UNIT}};line-height:{{SIZE}}{{UNIT}};' )
		        ->setSeparator( "after" )
		        ->getResponsive();

		$control->addTypography( 'typography_text', $btn )->getGroup();

		$this->start_controls_tabs( 'dsn_button_style' );

		$this->start_controls_tab(
			'tab_button_normal',
			[
				'label' => __( 'Normal', 'elementor' ),
			]
		);


		$control->addHeadingColor( 'button_text_color', $link_text )
		        ->setLabel( __( 'Text Color', 'blackdsn' ) )
		        ->setDefault( 'heading-color' )
		        ->get();


		$control->addColor( 'background_color' )
		        ->setLabel( __( 'Background Color', 'elementor' ) )
		        ->setSelectors( $btn, 'background-color: {{VALUE}};' )
		        ->get();


		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_button_hover',
			[
				'label' => __( 'Hover', 'elementor' ),
			]
		);


		$control->addColor( 'hover_color' )
		        ->setLabel( __( 'Text Color', 'elementor' ) )
		        ->setSelectors( $btn . ':hover ' . $link_text, 'color: {{VALUE}};' )
		        ->get();

		$control->addColor( 'button_background_hover_color' )
		        ->setLabel( __( 'Background Color', 'elementor' ) )
		        ->setSelectors( $btn . ':hover', 'background-color: {{VALUE}};' )
		        ->get();


		$control->addColor( 'button_hover_border_color' )
		        ->setLabel( __( 'Border Color', 'elementor' ) )
		        ->setSelectors( $btn . ':hover', 'border-color: {{VALUE}};' )
		        ->get();


		$this->end_controls_tab();

		$this->end_controls_tabs();


		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'      => 'border',
				'selector'  => '{{WRAPPER}} ' . $btn,
				'separator' => 'before',
			]
		);
		$control->addSelect( 'border_color_btn', $control->getBorderColor() )
		        ->setLabel( esc_html__( 'Border Color', 'blackdsn' ) )
		        ->setDefault( 'border-color-default' )
		        ->get();


		$this->add_control(
			'border_radius',
			[
				'label'      => __( 'Border Radius', 'elementor' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} ' . $btn => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'button_box_shadow',
				'selector' => '{{WRAPPER}} ' . $btn,
			]
		);

		$this->add_responsive_control(
			'text_padding',
			[
				'label'      => __( 'Padding', 'elementor' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} ' . $btn => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator'  => 'before',
			]
		);

		$this->end_controls_section();
		$this->styleIcon();
	}

	private function styleIcon() {
		$this->start_controls_section(
			'section_style_icon',
			[
				'label' => __( 'Icon', 'elementor' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$control = $this->getControl();


		$control->addIconColor( 'icon_color_select' )
		        ->setPrefix_class()
		        ->setDefault( 'dsn-icon-heading-color' )
		        ->get();


		$this->add_control(
			'icon_color',
			[
				'label'     => esc_html__( 'Color', 'elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'condition' => [
					'icon_color_select' => ''
				],
				'selectors' => [
					'{{WRAPPER}} .dsn-icon i'        => 'color: {{VALUE}};',
					'{{WRAPPER}} .dsn-icon svg'      => 'fill: {{VALUE}};',
					'{{WRAPPER}} .dsn-icon svg path' => 'fill: {{VALUE}};',
				],

			]
		);

		$control->addSlider( 'width_icon', $control->getDefaultWidthHeight() )
		        ->setLabel( __( 'Size Icon', 'elementor' ) )
		        ->setSelectors( '.dsn-btn .dsn-icon:not(.icon-link)', '--dsn-icon-size: {{SIZE}}{{UNIT}};' )
		        ->getResponsive();


		$control->addPaddingGroup( 'item_padding_icon', '.dsn-icon i ,{{WRAPPER}} .dsn-icon  svg' )
		        ->setSeparator( "before" )
		        ->getResponsive();


		$control->addBorderRadiusGroup( 'item_border_radius_icon', '.dsn-icon i ,{{WRAPPER}} .dsn-icon svg' )
		        ->getResponsive();

		$control->addBorder( 'item_border_style_icon', '.dsn-icon i ,{{WRAPPER}} .dsn-icon  svg' )->getGroup();
		$control->addBoxShadow( 'item_box_shadow_icon', '.dsn-icon i ,{{WRAPPER}} .dsn-icon  svg' )->getGroup();


		$this->end_controls_section();
	}


	/**
	 * Render accordion widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$this->add_inline_editing_attributes( 'text', 'none' );
		blackdsn_render_widget_motion_effect( $this, 'link' );

		echo blackdsn_shortcode_render_group( 'button', array( 'widget-base' => $this ) );
	}

	public function get_style_depends() {
		return $this->getDepends();
	}

	public function get_script_depends() {
		return $this->getDepends();
	}

	private function getDepends() {

		if ( blackdsn_is_built_with_elementor() ) {
			return [ blackdsnFrontEnd::FANCYBOX ];
		}
		$out       = [];
		$shortcode = $this->getShortCode();
		if ( $shortcode->getVal( 'popup_vid' ) ) {
			$out[] = blackdsnFrontEnd::FANCYBOX;
		}

		return $out;
	}


}
