<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}


/**
 * Elementor accordion widget.
 *
 * Elementor widget that displays a collapsible display of content in an
 * accordion style, showing only one item at a time.
 *
 * @since 1.0.0
 */
class  BlackdsnAnimateLine extends \Elementor\Widget_Base {

	use \Dsn\Element\Blackdsn_Widget_Base;


	/**
	 * Get widget name.
	 *
	 * Retrieve accordion widget name.
	 *
	 * @return string Widget name.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_name() {
		return 'dsn_animate_line';
	}


	/**
	 * Get widget title.
	 *
	 * Retrieve accordion widget title.
	 *
	 * @return string Widget title.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_title() {
		return __( 'Blackdsn Animate Line', 'blackdsn' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve accordion widget icon.
	 *
	 * @return string Widget icon.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_icon() {
		return 'eicon-t-letter';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @return array Widget keywords.
	 * @since 2.1.0
	 * @access public
	 *
	 */
	public function get_keywords() {
		return array_merge( $this->dsn_keywords(), [
			'information',
			'letter',
			'heading',
			'text',
			'paragraph',
			'animate',
			'line'
		] );
	}


	/**
	 * Register accordion widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {


		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Content', 'blackdsn' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);


		$control = $this->getControl();

		$control->addTextarea( 'title_before' )
		        ->setLabel( esc_html__( 'Title Before', 'blackdsn' ) )
		        ->setPlaceholder( esc_html__( 'My favourite food is', 'blackdsn' ) )
		        ->setDefault( esc_html__( 'Add Your Heading Text Here', 'blackdsn' ) )
		        ->get();
		$control->addTextarea( 'title_after' )
		        ->setLabel( esc_html__( 'Title After', 'blackdsn' ) )
		        ->get();

		$control->startRepeater();

		$control->addTextarea( 'text' )
		        ->setLabel( esc_html__( 'Text', 'blackdsn' ) )
		        ->get();

		$control->endRepeater( 'items' )
		        ->setLabel( esc_html__( "Item Text", 'blackdsn' ) )
		        ->setTitle_field( 'text' )
		        ->setSeparatorBefore()
		        ->setDefault( [
			        [ 'text' => 'pizza' ],
			        [ 'text' => 'sushi' ],
			        [ 'text' => 'steak' ],
		        ] )
		        ->get();


		$control->addSelect( 'animate_type', [
			'rotate-1'           => esc_html__( 'Full Rotate ', 'blackdsn' ),
			'letters rotate-2'   => esc_html__( 'Letters Rotate Y', 'blackdsn' ),
			'letters rotate-3'   => esc_html__( 'Letters Rotate X', 'blackdsn' ),
			'letters type'       => esc_html__( 'Letters Type', 'blackdsn' ),
			'loading-bar'        => esc_html__( 'Loading Bar', 'blackdsn' ),
			'slide'              => esc_html__( 'Slide', 'blackdsn' ),
			'clip is-full-width' => esc_html__( 'Clip', 'blackdsn' ),
			'zoom'               => esc_html__( 'Zoom', 'blackdsn' ),
			'letters scale'      => esc_html__( 'Letters Scale', 'blackdsn' ),
			'push'               => esc_html__( 'Push', 'blackdsn' ),

		] )
		        ->setLabel( esc_html__( 'Animate Type', 'blackdsn' ) )
		        ->setDefault( 'rotate-1' )
		        ->get();

		$control->addSize()
		        ->setSeparator( "before" )
		        ->get();

		$control->addLineText()->get();

		$control->addHtmlTag()->get();
		$control->getAlign()->getResponsive();


		$this->end_controls_section();

		$this->styleTab();
		$this->TitleCover();

	}


	private function styleTab() {
		$control = $this->getControl();

		$this->start_controls_section(
			'style_section',
			[
				'label' => esc_html__( 'Style', 'blackdsn' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$control->addImage()
		        ->setDefault( [] )
		        ->setSelectors( '.dsn-heading-title', 'background-image: url("{{URL}}");color: transparent;-webkit-text-stroke: 3.5px var(--heading-color);-webkit-text-fill-color: transparent;-webkit-background-clip: text;background-clip: text;transform: scale3d(1, 1.2, 1);background-position: center;
background-repeat: no-repeat;
background-size: cover;' )
		        ->getResponsive();

		$control->addHeadingColor( 'title_color', '.dsn-heading-title' )
		        ->setLabel( __( 'Custom Color', 'blackdsn' ) )
		        ->get();

		$control->addSlider( 'space', $control->getDefaultWidthHeight() )
		        ->setLabel( __( 'Max Width', 'elementor' ) )
		        ->setSelectors( '.dsn-heading-title', 'max-width: {{SIZE}}{{UNIT}};' )
		        ->getResponsive();

		$control->addTypography( 'title_typography' )->getGroup();
		$control->addTextShadow( 'title_Text_shadow' )->getGroup();
		$control->addBlendMode( 'title_blend_mode' )->get();

		$control->addSwitcher( 'use_as_troke' )
		        ->setReturn_value( 'letter-stroke' )
		        ->setLabel( __( 'Use As Stroke', 'blackdsn' ) )
		        ->get();


		$this->end_controls_section();


	}

	private function TitleCover() {
		$control = $this->getControl();

		$this->start_controls_section(
			'section_title_cover',
			[
				'label' => esc_html__( 'Title Cover Option', 'blackdsn' ),
				'tab'   => \Elementor\Controls_Manager::TAB_SETTINGS,
			]
		);

		$control->addSwitcher( 'use_as_title_cover' )
		        ->setLabel( __( 'Use As Title cover', 'blackdsn' ) )
		        ->setReturn_value( '1' )
		        ->get();

		$control->addHiddenNoSpace( 'hidden-space' )
		        ->setConditions( 'use_as_title_cover', '1' )
		        ->get();


		$control->addNumber( 'optcity_from', 0, 1, 0.1 )
		        ->setLabel( __( 'Optcity From', 'blackdsn' ) )
		        ->setDefault( 0 )
		        ->setSelectors( '.dsn-move-section', 'opacity : {{VALUE}}' )
		        ->setConditions( 'use_as_title_cover', '1' )
		        ->get();


		$control->addNumber( 'optcity_to', 0, 1, 0.1 )
		        ->setLabel( __( 'Optcity To', 'blackdsn' ) )
		        ->setDefault( 0.07 )
		        ->setConditions( 'use_as_title_cover', '1' )
		        ->get();


		$control->addText( 'move_from_duration' )
		        ->setLabel( __( 'Move From', 'blackdsn' ), true )
		        ->setDefault( '-100%' )
		        ->setDescription( __( 'Move Section From.', 'blackdsn' ) )
		        ->setConditions( 'use_as_title_cover', '1' )
		        ->setSeparatorBefore()
		        ->get();

		$control->addText( 'move_to_duration' )
		        ->setLabel( __( 'Move To', 'blackdsn' ), true )
		        ->setDefault( '0%' )
		        ->setDescription( __( 'Move Section To.', 'blackdsn' ) )
		        ->setConditions( 'use_as_title_cover', '1' )
		        ->get();


		$control->getTriggerHock( 'triggerhook_duration' )
		        ->setConditions( 'use_as_title_cover', '1' )
		        ->setSeparatorBefore()
		        ->get();

		$control->addSwitcher( 'stop_on_mobile' )
		        ->setLabel( __( 'Stop On Mobile', 'blackdsn' ) )
		        ->setReturn_value( '1' )
		        ->setConditions( 'use_as_title_cover', '1' )
		        ->get();


		$this->end_controls_section();
	}

	/**
	 * Render accordion widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {

		$this->add_inline_editing_attributes( 'title_before' );
		$this->add_inline_editing_attributes( '$title_after' );
		echo blackdsn_shortcode_render_group( 'animate-line', array( 'widget-base' => $this ) );

	}

	public function get_style_depends(): array {
		return $this->getDepends();
	}

	public function get_script_depends(): array {
		return $this->getDepends();
	}

	private function getDepends(): array {
		return [ blackdsnFrontEnd::ANIMATED_HEADLINE ];
	}
//	protected function content_template() {
//		echo blackdsn_shortcode_js_group( 'heading', array(), '' );
//	}


}
