<?php

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}


/**
 * Elementor accordion widget.
 *
 * Elementor widget that displays a collapsible display of content in an
 * accordion style, showing only one item at a time.
 *
 * @since 1.0.0
 */
class BlackdsnAccordion extends \Elementor\Widget_Base
{

    use \Dsn\Element\Blackdsn_Widget_Base;


    /**
     * Get widget name.
     *
     * Retrieve accordion widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'dsn_accordion';
    }


    /**
     * Get widget title.
     *
     * Retrieve accordion widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return __('Blackdsn Accordion', 'blackdsn');
    }

    /**
     * Get widget icon.
     *
     * Retrieve accordion widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-accordion';

    }

    /**
     * Get widget keywords.
     *
     * Retrieve the list of keywords the widget belongs to.
     *
     * @return array Widget keywords.
     * @since 2.1.0
     * @access public
     *
     */
    public function get_keywords()
    {
        return array_merge($this->dsn_keywords(), ['accordion', 'tabs', 'toggle']);
    }


    /**
     * Register accordion widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {


        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__('Accordion', 'blackdsn'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );


        $control = $this->getControl();

        $control->addSwitcher('list_with_number')
            ->setLabel(esc_html__("List With Number", 'blackdsn'))
            ->setDescription(esc_html__("if Us this feature can't us icon with list"))
            ->get();

        $control->startRepeater();

        $control->get_element_base()->add_control(
            'icon',
            [
                'label' => esc_html__('Icon', 'elementor'),
                'type' => \Elementor\Controls_Manager::ICONS,
            ]
        );

        $control->addText('title')
            ->setDefault('Web Design')
            ->setLabel(esc_html__("Title", 'blackdsn'))
            ->setDynamicActive(true)
            ->get();

        $control->addTextareaEditor('description')
            ->setLabel(esc_html__("Description", 'blackdsn'))
            ->setDefault('<p>Cepteur sint occaecat cupidatat proident, taken possession of my entire soul, like these sweet mornings of spring which I enjoy with my whole.</p>')
            ->setTitle_field('title')
            ->setDynamicActive(true)
            ->get();

        $control->endRepeater('items')
            ->setLabel(esc_html__('Accordion Items', 'blackdsn'))
            ->setTitle_field_withIcon('icon', 'title')
            ->get();

        $control->addHtmlTag()
            ->setSeparator("before")
            ->get();

        $control->addSize()
            ->setDefault('sm-title-block')
            ->get();


        $this->end_controls_section();

        $this->styleTab();

    }


    private function styleTab()
    {
        $control = $this->getControl();

        $this->start_controls_section(
            'style_section',
            [
                'label' => esc_html__('Style', 'blackdsn'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->__style_controller($control);

        $this->end_controls_section();
        $this->__style_content_controller($control);

    }

    public function __style_controller(BlackdsnControl $control)
    {


        $control->addSelect('bg_ver_btn', $control->getOptionVerBackground())
            ->setLabel(esc_html__('Version Background', 'blackdsn'))
            ->setLabelBlock()
            ->setDefault('')
            ->setPrefix_class()
            ->get();

        $control->addSelect('bg_btn', $control->getOptionBackground())
            ->setLabel(esc_html__('Background', 'blackdsn'))
            ->setLabelBlock()
            ->setDefault('background-transparent')
            ->setPrefix_class()
            ->get();

        $control->addPaddingGroup('item_padding_service', '.dsn-accordion')
            ->setSeparator("before")
            ->getResponsive();


        $control->addBorderRadiusGroup('item_border_radius', '')
            ->getResponsive();

        $control->addBorder('item_border_style', '.dsn-accordion')->getGroup();
        $control->addBoxShadow('item_box_shadow', '.dsn-accordion')->getGroup();
    }
    public function __style_content_controller( BlackdsnControl $control ) {
        $args = array(
            'tab-title'       => esc_html__( 'Title', 'blackdsn' ),
            'tab-description' => esc_html__( 'Description', 'blackdsn' )
        );

        foreach ( $args as $id => $value ):

            $this->start_controls_section(
                'style_content_service_' . $id,
                [
                    'label' => $value,
                    'tab'   => Controls_Manager::TAB_STYLE,
                ]
            );


            $control->addColor( 'color_content_' . $id )
                ->setLabel( esc_html__( "Color", 'blackdsn' ) )
                ->setSeparator( "before" )
                ->setSelectors( '.' . $id, 'color:{{VALUE}};' )
                ->get();

            $control->addTypography( 'item_typo_content_' . $id, '.' . $id )
                ->getGroup();


            $this->end_controls_section();


        endforeach;


        $this->start_controls_section(
            'style_content_service_icon',
            [
                'label' => esc_html__( "icon", 'blackdsn' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );


        $control->addIconColor( 'icon_color_select' )
            ->setPrefix_class()
            ->get();


        $this->add_control(
            'icon_color',
            [
                'label'     => esc_html__( 'Color', 'elementor' ),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'condition' => [
                    'icon_color_select' => ''
                ],
                'selectors' => [
                    '{{WRAPPER}} .dsn-icon i'        => 'color: {{VALUE}};',
                    '{{WRAPPER}} .dsn-icon svg'      => 'fill: {{VALUE}};',
                    '{{WRAPPER}} .dsn-icon svg path' => 'fill: {{VALUE}};',
                ],

            ]
        );

        $control->addSlider( 'width_icon', $control->getDefaultWidthHeight() )
            ->setLabel( __( 'Size Icon', 'elementor' ) )
            ->setSelectors( '.dsn-accordion .dsn-icon:not(.icon-link)', '--dsn-icon-size: {{SIZE}}{{UNIT}};' )
            ->getResponsive();

        $control->addSlider( 'icon_indent' )
            ->setLabel( __( 'Icon Spacing', 'elementor' ) )
            ->setRangePx( 0, 100 )
            ->setSelectors( '.dsn-icon', 'margin-right: {{SIZE}}{{UNIT}};' )
            ->get();

        $control->addPaddingGroup( 'item_padding_icon', '.dsn-icon i ,{{WRAPPER}} .dsn-icon  svg' )
            ->setSeparator( "before" )
            ->getResponsive();


        $control->addBorderRadiusGroup( 'item_border_radius_icon', '.dsn-icon i ,{{WRAPPER}} .dsn-icon svg' )
            ->getResponsive();

        $control->addBorder( 'item_border_style_icon', '.dsn-icon i ,{{WRAPPER}} .dsn-icon  svg' )->getGroup();
        $control->addBoxShadow( 'item_box_shadow_icon', '.dsn-icon i ,{{WRAPPER}} .dsn-icon  svg' )->getGroup();

        $this->end_controls_section();
    }

    /**
     * Render accordion widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {

        $items_key = $this->getKeys('items', ['title', 'description' => 'advanced']);
        echo blackdsn_shortcode_render_group('accordion', array('widget-base' => $this), $items_key);

    }


}
