<?php


use Elementor\Controls_Manager;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use Elementor\Group_Control_Typography;
use Elementor\Repeater;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}


/**
 * Elementor accordion widget.
 *
 * Elementor widget that displays a collapsible display of content in an
 * accordion style, showing only one item at a time.
 *
 * @since 1.0.0
 */
class ArctitPricingTable extends \Elementor\Widget_Base {

	use \Dsn\Element\Arctit_Widget_Base;


	/**
	 * Get widget name.
	 *
	 * Retrieve accordion widget name.
	 *
	 * @return string Widget name.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_name() {
		return 'dsn_pricing_table';
	}


	/**
	 * Get widget title.
	 *
	 * Retrieve accordion widget title.
	 *
	 * @return string Widget title.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_title() {
		return __( 'Arctit Pricing Table', 'arctit' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve accordion widget icon.
	 *
	 * @return string Widget icon.
	 * @since 1.0.0
	 * @access public
	 *
	 */
	public function get_icon() {
		return 'eicon-price-table';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @return array Widget keywords.
	 * @since 2.1.0
	 * @access public
	 *
	 */
	public function get_keywords() {
		return array_merge( $this->dsn_keywords(), [ 'pricing', 'table', 'product', 'image', 'plan', 'button' ] );
	}


	/**
	 * Register accordion widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {


		$this->heading();
		$this->pricing();
		$this->feature();
		$this->button();


	}


	private function styleItem() {
		$control = $this->getControl();

		$this->start_controls_section(
			'section_item_style',
			[
				'label'      => __( 'Card', 'arctit' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			]
		);
		$control->addSelect( 'background_ver_mask', $control->getOptionVerBackground() )
		        ->setLabel( esc_html__( 'Version Background', 'arctit' ) )
		        ->setDefault( '' )
		        ->get();

		$control->addSelect( 'background_mask', $control->getOptionBackground() )
		        ->setLabel( esc_html__( 'Background Mask', 'arctit' ) )
		        ->setDefault( 'background-transparent' )
		        ->get();




		$this->add_group_control(
			\Elementor\Group_Control_Background::get_type(),
			[
				'name'     => 'background',
				'label'    => esc_html__( 'Background', 'arctit' ),
				'types'    => [ 'classic', 'gradient', 'video' ],
				'selector' => '{{WRAPPER}} .dsn-bg-mask',
			]
		);

		$this->add_responsive_control(
			'opacity',
			[
				'label'     => __( 'Opacity', 'elementor' ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'max'  => 1,
						'min'  => 0.10,
						'step' => 0.01,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .dsn-bg-mask ' => 'opacity: {{SIZE}};',
				],
			]
		);

		$control->addPaddingGroup( 'card_padding', '.dsn-price-table' )
			->setSeparatorBefore()
		        ->getResponsive();

		$control->addBorder( 'item_border_card', '.dsn-price-table' )->getGroup();





		$this->add_responsive_control(
			'image_border_radius',
			[
				'label'      => __( 'Border Radius', 'elementor' ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'separator'  => 'before',
				'selectors'  => [
					'{{WRAPPER}} .dsn-bg-mask  ,  {{WRAPPER}} .dsn-price-table' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Css_Filter::get_type(),
			[
				'name'     => 'css_filters',
				'selector' => '{{WRAPPER}} .dsn-bg-mask ',
			]
		);


		$this->end_controls_section();
	}

	private function heading() {
		$control = $this->getControl();

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Header', 'arctit' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);


		$control->addText( 'title' )
		        ->setLabel( esc_html__( "Title", 'arctit' ) )
		        ->setDefault( esc_html__( "Enter your title", 'arctit' ) )
		        ->get();

		$control->addText( 'description' )
		        ->setLabel( esc_html__( "SubTitle", 'arctit' ) )
		        ->setDefault( esc_html__( "Enter your description", 'arctit' ) )
		        ->get();

		$control->addHtmlTag()->setDefault( 'h3' )->get();

		$this->end_controls_section();

		$this->styleItem();
		/**
		 * Style
		 */
		$this->start_controls_section(
			'section_header_style',
			[
				'label'      => __( 'Header', 'arctit' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			]
		);

		$this->add_control(
			'header_bg_color',
			[
				'label'     => __( 'Background Color', 'arctit' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .price-header' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'header_padding',
			[
				'label'      => __( 'Padding', 'arctit' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .price-header' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'heading_heading_style',
			[
				'label'     => __( 'Title', 'arctit' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'heading_color',
			[
				'label'     => __( 'Color', 'arctit' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .title-heading' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'heading_typography',
				'selector' => '{{WRAPPER}} .title-heading',
			]
		);

		$this->add_control(
			'heading_sub_heading_style',
			[
				'label'     => __( 'Sub Title', 'arctit' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'sub_heading_color',
			[
				'label'     => __( 'Color', 'arctit' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .subtitle-heading' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'sub_heading_typography',
				'selector' => '{{WRAPPER}} .subtitle-heading',

			]
		);

		$this->end_controls_section();
	}

	private function pricing() {
		$control = $this->getControl();

		$this->start_controls_section(
			'content_price_section',
			[
				'label' => esc_html__( 'Pricing', 'arctit' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);


		$control->addSelect( 'currency_symbol', [
			''             => __( 'None', 'arctit' ),
			'dollar'       => '&#36; ' . esc_html__( 'Dollar', 'arctit' ),
			'euro'         => '&#128; ' . esc_html__( 'Euro', 'arctit' ),
			'baht'         => '&#3647; ' . esc_html__( 'Baht', 'arctit' ),
			'franc'        => '&#8355; ' . esc_html__( 'Franc', 'arctit' ),
			'guilder'      => '&fnof; ' . esc_html__( 'Guilder', 'arctit' ),
			'krona'        => 'kr ' . esc_html__( 'Krona', 'arctit' ),
			'lira'         => '&#8356; ' . esc_html__( 'Lira', 'arctit' ),
			'peseta'       => '&#8359 ' . esc_html__( 'Peseta', 'arctit' ),
			'peso'         => '&#8369; ' . esc_html__( 'Peso', 'arctit' ),
			'pound'        => '&#163; ' . esc_html__( 'Pound Sterling', 'arctit' ),
			'real'         => 'R$ ' . esc_html__( 'Real', 'arctit' ),
			'ruble'        => '&#8381; ' . esc_html__( 'Ruble', 'arctit' ),
			'rupee'        => '&#8360; ' . esc_html__( 'Rupee', 'arctit' ),
			'indian_rupee' => '&#8377; ' . esc_html__( 'Rupee (Indian)', 'arctit' ),
			'shekel'       => '&#8362; ' . esc_html__( 'Shekel', 'arctit' ),
			'yen'          => '&#165; ' . esc_html__( 'Yen/Yuan', 'arctit' ),
			'won'          => '&#8361; ' . esc_html__( 'Won', 'arctit' ),
			'custom'       => __( 'Custom', 'arctit' ),
		] )
		        ->setLabel( __( 'Currency Symbol', 'arctit' ) )
		        ->setDefault( "dollar" )
		        ->get();


		$control->addText( 'currency_symbol_custom' )
		        ->setLabel( __( 'Custom Symbol', 'arctit' ) )
		        ->setConditions( 'currency_symbol', 'custom' )
		        ->get();

		$control->addText( 'price' )
		        ->setLabel( __( 'Price', 'arctit' ) )
		        ->setDefault( "30" )
		        ->get();

		$control->addText( 'period' )
		        ->setLabel( __( 'Period', 'arctit' ) )
		        ->setDefault( __( '/per mo', 'arctit' ) )
		        ->get();


		$control->addSwitcher( 'sale' )
		        ->setLabel( __( 'Sale', 'arctit' ) )
		        ->get();

		$control->addNumber( 'original_price', null, null, 1 )
		        ->setLabel( __( 'Original Price', 'arctit' ) )
		        ->setDefault( "50" )
		        ->setConditions( 'sale', '1' )
		        ->get();

		$control->addText( 'text_original_price' )
		        ->setLabel( __( 'Text Before', 'arctit' ) )
		        ->setDefault( __( 'Normally', 'arctit' ) )
		        ->setConditions( 'sale', '1' )
		        ->get();


		$control->addTextarea( 'description_price' )
		        ->setLabel( __( 'Description', 'arctit' ) )
		        ->setDefault( __( 'Lorem ipsum dolor sit amdoet, consectetur adipiscing elitde us uodtempor incidminim.', 'arctit' ) )
		        ->setSeparatorBefore()
		        ->get();

		$this->end_controls_section();


		$this->start_controls_section(
			'section_pricing_element_style',
			[
				'label' => __( 'Pricing', 'arctit' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'pricing_element_bg_color',
			[
				'label'     => __( 'Background Color', 'arctit' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .price-table-price' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'pricing_element_padding',
			[
				'label'      => __( 'Padding', 'arctit' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .price-table-price' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'price_color',
			[
				'label'     => __( 'Color', 'arctit' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .price-currency, {{WRAPPER}} .price-int-part' => 'color: {{VALUE}}',
				],
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'price_typography',
				'selector' => '{{WRAPPER}} .price-table-price .title',
			]
		);

		$this->add_control(
			'heading_currency_style',
			[
				'label'     => __( 'Currency Symbol', 'arctit' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'currency_symbol!' => '',
				],
			]
		);


		$this->add_responsive_control(
			'currency_size',
			[
				'label'     => __( 'Size', 'arctit' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .price-currency' => 'font-size:{{SIZE}}px',
				],
				'condition' => [
					'currency_symbol!' => '',
				],
			]
		);


		$this->add_responsive_control(
			'currency_vertical_position',
			[
				'label'                => __( 'Vertical Position', 'arctit' ),
				'type'                 => Controls_Manager::CHOOSE,
				'options'              => [
					'top'    => [
						'title' => __( 'Top', 'arctit' ),
						'icon'  => 'eicon-v-align-top',
					],
					'middle' => [
						'title' => __( 'Middle', 'arctit' ),
						'icon'  => 'eicon-v-align-middle',
					],
					'bottom' => [
						'title' => __( 'Bottom', 'arctit' ),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'default'              => 'top',
				'selectors_dictionary' => [
					'top'    => 'flex-start',
					'middle' => 'center',
					'bottom' => 'flex-end',
				],
				'selectors'            => [
					'{{WRAPPER}} .price-currency' => 'align-self: {{VALUE}}',
				],
				'condition'            => [
					'currency_symbol!' => '',
				],
			]
		);


		$this->add_control(
			'heading_original_price_style',
			[
				'label'     => __( 'Original Price', 'arctit' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'sale'            => '1',
					'original_price!' => '',
				],
			]
		);

		$this->add_control(
			'original_price_color',
			[
				'label'     => __( 'Color', 'arctit' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .price-original-price' => 'color: {{VALUE}}',
				],
				'condition' => [
					'sale'            => '1',
					'original_price!' => '',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'      => 'original_price_typography',
				'selector'  => '{{WRAPPER}} .price-original-price',
				'condition' => [
					'sale'            => '1',
					'original_price!' => '',
				],
			]
		);


		$this->add_control(
			'heading_period_style',
			[
				'label'     => __( 'Period', 'arctit' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'period!' => '',
				],
			]
		);

		$this->add_control(
			'period_color',
			[
				'label'     => __( 'Color', 'arctit' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .price-period' => 'color: {{VALUE}}',
				],
				'condition' => [
					'period!' => '',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'      => 'period_typography',
				'selector'  => '{{WRAPPER}} .price-period',
				'condition' => [
					'period!' => '',
				],
			]
		);

		$this->add_responsive_control(
			'period_vertical_position',
			[
				'label'                => __( 'Vertical Position', 'arctit' ),
				'type'                 => Controls_Manager::CHOOSE,
				'options'              => [
					'top'    => [
						'title' => __( 'Top', 'arctit' ),
						'icon'  => 'eicon-v-align-top',
					],
					'middle' => [
						'title' => __( 'Middle', 'arctit' ),
						'icon'  => 'eicon-v-align-middle',
					],
					'bottom' => [
						'title' => __( 'Bottom', 'arctit' ),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'default'              => 'bottom',
				'selectors_dictionary' => [
					'top'    => 'flex-start',
					'middle' => 'center',
					'bottom' => 'flex-end',
				],
				'selectors'            => [
					'{{WRAPPER}} .price-period' => 'align-self: {{VALUE}}',
				],
				'condition'            => [
					'currency_symbol!' => '',
				],
			]
		);

		$this->end_controls_section();

	}

	private function feature() {
		$control = $this->getControl();

		$this->start_controls_section(
			'content_feature_section',
			[
				'label' => esc_html__( 'Features', 'arctit' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);


		$repeater = new Repeater();

		$repeater->add_control(
			'text',
			[
				'label'   => __( 'Text', 'arctit' ),
				'type'    => Controls_Manager::TEXT,
				'default' => __( 'List Item', 'arctit' ),
			]
		);

		$default_icon = [
			'value'   => 'far fa-check-circle',
			'library' => 'fa-regular',
		];

		$repeater->add_control(
			'icon',
			[
				'label'            => __( 'Icon', 'arctit' ),
				'type'             => Controls_Manager::ICONS,
				'fa4compatibility' => 'item_icon',
				'default'          => $default_icon,
			]
		);

		$repeater->add_control(
			'item_icon_color',
			[
				'label'     => __( 'Icon Color', 'arctit' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} {{CURRENT_ITEM}} i'   => 'color: {{VALUE}}',
					'{{WRAPPER}} {{CURRENT_ITEM}} svg' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'items',
			[
				'type'    => Controls_Manager::REPEATER,
				'fields'  => $repeater->get_controls(),
				'default' => [
					[
						'text' => __( 'List Item #1', 'arctit' ),
						'icon' => $default_icon,
					],
					[
						'text' => __( 'List Item #2', 'arctit' ),
						'icon' => $default_icon,
					],
					[
						'text' => __( 'List Item #3', 'arctit' ),
						'icon' => $default_icon,
					],
				],

				'title_field' => '{{{elementor.helpers.renderIcon( this, icon, {}, "i", "panel" ) || \'<i class="{{ icon }}" aria-hidden="true"></i>\'}}} {{{ text }}}',
			]
		);


		$this->end_controls_section();

		/**
		 * Style
		 */

		$this->start_controls_section(
			'style_feature_section',
			[
				'label' => esc_html__( 'Style', 'arctit' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'feature_element_bg_color',
			[
				'label'     => __( 'Background Color', 'arctit' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .price-feature' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'feature_element_padding',
			[
				'label'      => __( 'Padding', 'arctit' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .price-feature' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$control->addSlider( 'gaps' )
		        ->setLabel( __( 'Gaps', 'elementor' ) )
		        ->setRangePx( 0, 100 )
		        ->setSelectors( 'li:not(:last-of-type)', 'margin-bottom: {{SIZE}}{{UNIT}};' )
		        ->setSeparatorBefore()
		        ->getResponsive();


		$control->addHeadingColor( 'title_color', '.dsn-heading-title' )
		        ->setLabel( __( 'Custom Color', 'arctit' ) )
		        ->get();


		$control->addTypography( 'title_typography' )->getGroup();
		$control->addTextShadow( 'title_Text_shadow' )->getGroup();


		$control->addSlider( 'icon_indent' )
		        ->setLabel( __( 'Icon Spacing', 'elementor' ) )
		        ->setRangePx( 0, 50 )
		        ->setSelectors( '.dsn-icon', 'margin-right: {{SIZE}}{{UNIT}};' )
		        ->getResponsive();

		$this->add_responsive_control(
			'icon_size',
			[
				'label'     => esc_html__( 'Size', 'elementor' ),
				'type'      => Controls_Manager::SLIDER,
				'default'   => [
					'size' => 16,
				],
				'range'     => [
					'px' => [
						'min' => 6,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .dsn-icon' => '--dsn-icon-size: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();


	}

	private function button() {
		$control = $this->getControl();

		$this->start_controls_section(
			'content_footer_section',
			[
				'label' => esc_html__( 'Footer', 'arctit' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);


		$control->addText( "text" )
		        ->setLabel( __( 'Text', 'elementor' ) )
		        ->setDynamicActive( true )
		        ->setDefault( __( 'Click here', 'elementor' ) )
		        ->setPlaceholder( __( 'Click here', 'elementor' ) )
		        ->get();

		$control->addLink( 'link' )
		        ->setLabel( esc_html__( 'Link', 'arctit' ) )
		        ->setDefault_url( '#0' )
		        ->setDynamicActive( true )
		        ->setCondition( [ 'text!' => '' ] )
		        ->get();

		$this->end_controls_section();

		/**
		 * Style
		 */
		$this->start_controls_section(
			'section_button_style',
			[
				'label'      => __( 'Footer', 'arctit' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			]
		);


		$this->add_control(
			'footer_element_bg_color',
			[
				'label'     => __( 'Background Color', 'arctit' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .price-footer' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'footer_element_padding',
			[
				'label'      => __( 'Padding', 'arctit' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .price-footer' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$control->addTypography( 'footer_typography', '.dsn-post-link' )
		        ->setSeparatorBefore()
		        ->getGroup();
		$control->addTextShadow( 'footer_Text_shadow', '.dsn-post-link' )->getGroup();


		$this->end_controls_section();
	}

	/**
	 * Render accordion widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		//--> Head
		$this->add_inline_editing_attributes( 'title' );
		$this->add_inline_editing_attributes( 'description' );
		//--> price
		$this->add_inline_editing_attributes( 'original_price' );
		$this->add_inline_editing_attributes( 'period' );
		$this->add_inline_editing_attributes( 'price' );
		$this->add_inline_editing_attributes( 'description_price' );
		$this->add_inline_editing_attributes( 'text' );


		$this->getKeys( 'items', [ 'text' ] );

		echo arctit_shortcode_render_group( 'price-table', array( 'widget-base' => $this ) );
	}

}
